#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Asks the user whether to uninstall the Homebrew. If so, then the Homebrew is
# uninstalled.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"



# Check if Homebrew is installed.
command -v brew &>/dev/null
local homebrew_is_installed=$?
if [ ${homebrew_is_installed} -eq 0 ]
then
    echo "Homebrew found."
else
    echo "Homebrew is not installed."
fi



# If Homebrew is installed, then check if Wine is also installed.
local wine_command_available=1
if [ ${homebrew_is_installed} -eq 0 ]
then
    command -v wine &>/dev/null
    local wine_command_available=$?
    if [ ${wine_command_available} -eq 0 ]
    then
        echo "Wine found."
    fi
fi



# If Homebrew is installed, then ask the user whether it should be uninstalled.
local uninstall_homebrew=1
if [ ${homebrew_is_installed} -eq 0 ] && [ ${force_operation} -ne 0 ]
then
    echo ${warning_style}"Uninstalling Homebrew will prevent the use of their packages on macOS."${default_style}
    echo ${warning_style}"If you have other packages installed through Homebrew that are being used,"${default_style}
    echo ${warning_style}"then this is not recommended."${default_style}
    if [ ${wine_command_available} -eq 0 ]
    then
        echo ${warning_style}"It appears that Wine is installed. A Wine installation is typically managed"${default_style}
        echo ${warning_style}"by Homebrew."${default_style}
    fi
    prompt_user_default_no ${input_style}"Should Homebrew be uninstalled? [y/N] or press CTRL-C to abort."${default_style}
    local uninstall_homebrew=$?
fi



# Uninstall Homebrew if so desired.
if [ ${uninstall_homebrew} -eq 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Uninstalling Homebrew."
    /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/uninstall.sh)"
else
    echo "Homebrew is NOT being uninstalled."
fi



# Check if Homebrew has been uninstalled and "return" 0 (true) or 1 (false).
command -v brew &>/dev/null
local homebrew_is_installed=$?
if [ ${uninstall_homebrew} -eq 0 ] && [ ${homebrew_is_installed} -eq 0 ]
then
    echo ${error_style}"FAILED uninstalling Homebrew."
    echo "Please uninstall Homebrew manually."${default_style}
    return 1
else
    return 0
fi
