#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Asks the user whether to uninstall the Wine. If so, then the Wine is
# uninstalled.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"



# Check if Wine is installed.
command -v wine &>/dev/null
local wine_command_available=$?
if [ ${wine_command_available} -eq 0 ]
then
    echo "Wine found."
else
    echo "Wine is not installed."
fi



# If Wine is installed, then check if Homebrew is also installed.
if [ ${wine_command_available} -eq 0 ]
then
    command -v brew &>/dev/null
    local homebrew_is_installed=$?
    # If Homebrew is not installed, then consider Wine not being installed for our purposes.
    if [ ${homebrew_is_installed} -ne 0 ]
    then
        echo "Homebrew not installed. It is needed to un-/install Wine."
        local wine_command_available=1
    fi
fi



# If Wine is installed, then ask the user whether it should be uninstalled.
local uninstall_wine=1
if [ ${wine_command_available} -eq 0 ] && [ ${force_operation} -ne 0 ]
then
    echo ${warning_style}"Uninstalling Wine will prevent the use of other ADOxx based tools and other"${default_style}
    echo ${warning_style}"(Windows) applications that rely on Wine. If you have other ADOxx based"${default_style}
    echo ${warning_style}"tools that you want to use, then this is not recommended."${default_style}
    prompt_user_default_no ${input_style}"Should Wine be uninstalled? [y/N] or press CTRL-C to abort."${default_style}
    local uninstall_wine=$?
fi



# Uninstall Wine if so desired.
if [ ${uninstall_wine} -eq 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Uninstalling Wine."
    # Using wine-staging / wine@staging, because using wine-stable has sometimes caused an
    # error during prefix creation like:
    # wine: could not open working directory L"C:\\windows\\system32\\", starting in the Windows directory
    #brew uninstall --cask wine-stable
    # This was used until 2024-12
    #brew uninstall --cask wine-staging
    brew uninstall --cask wine@staging
else
    echo "Wine is NOT being uninstalled."
fi



# Check if Wine has been uninstalled and "return" 0 (true) or 1 (false).
command -v wine &>/dev/null
local wine_command_available=$?
if [ ${uninstall_wine} -eq 0 ] && [ ${wine_command_available} -eq 0 ]
then
    echo ${error_style}"FAILED uninstalling Wine."
    echo "Please uninstall Wine manually."${default_style}
    return 1
else
    return 0
fi
