#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Asks the user whether to delete the Wine prefix. If so, then the Wine prefix
# folder with all its contents is deleted.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.
# wine_prefix_dir - The Wine prefix directory.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"



# Check if the Wine prefix exists.
[ -d "${wine_prefix_dir}" ] &>/dev/null
local wine_prefix_exists=$?
if [ ${wine_prefix_exists} -eq 0 ]
then
    echo "Wine prefix found: ${wine_prefix_dir}"
else
    echo "Wine prefix is missing."
fi



# If the Wine prefix exists, then ask the user whether it should be deleted.
local delete_wineprefix=1
if [ ${wine_prefix_exists} -eq 0 ] && [ ${force_operation} -ne 0 ]
then
    echo ${warning_style}"Deleting the Wine prefix will delete all ADOxx based tools and any other"${default_style}
    echo ${warning_style}"(Windows) applications installed in that prefix. If you have other ADOxx"${default_style}
    echo ${warning_style}"based tools that you want to use, then this is not recommended."${default_style}
    echo ${warning_style}"Deleting the Wine prefix also deletes the database file and all models therein."${default_style}
    echo ${warning_style}"Make sure you no longer need them or that they have been exported as"${default_style}
    echo ${warning_style}".adl / .xml files first."${default_style}
    prompt_user_default_no ${input_style}"Should the Wine prefix be deleted? [y/N] or press CTRL-C to abort."${default_style}
    local delete_wineprefix=$?
fi



# Delete the Wine prefix if so desired.
if [ ${delete_wineprefix} -eq 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Deleting Wine prefix: ${wine_prefix_dir}"
    rm -rf "${wine_prefix_dir}"
else
    echo "Wine prefix is NOT being deleted."
fi



# Check if the Wine prefix has been removed and "return" 0 (true) or 1 (false).
if [ ${delete_wineprefix} -eq 0 ] && [ -d "${wine_prefix_dir}" ]
then
    echo ${error_style}"FAILED deleting Wine prefix."
    echo "Please delete the Wine prefix manually."${default_style}
    return 1
else
    return 0
fi
