#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Asks the user whether to delete the database file. If so, then the database
# file is deleted. This includes all the models that have been stored there.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.
# wine_prefix_dir - The Wine prefix directory.
# database_name - The name of the database, which is also the name of the file.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"



# Check if the database already exist.
# The logic has to follow the same logic that is used by ADOxx.
# The first test is SIMPLE, assuming that the file is in a very specific
# location. This SHOULD be the most likely case.
local user_folder="$(id -un)"
local database_file="${wine_prefix_dir}/drive_c/users/${user_folder}/AppData/Roaming/ADOxx/sqlitedbs/${database_name}.sqlite3"
[ -f "${database_file}" ]
local database_exists=$?
if [ ${database_exists} -eq 0 ]
then
    echo "Database file found: ${database_file}"
fi

# If the previous check did not find the database file, then we "ask" Wine
# instead. There are cases where the file can be located in a different place:
# * Some Wine versions use "Application Data/" instead "AppData/Roaming/".
# * Some Wine versions do not use the system user's name for the user name
#   in the Wine prefix.
if [ ${database_exists} -ne 0 ]
then
    # First check if the Wine prefix folder exists and Wine is installed.
    if [ ! -d "${wine_prefix_dir}" ]
    then
        echo ${warning_style}"Wine prefix not available. So the database is most likely gone too."${default_style}
        exit 0
    fi
    command -v wine &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ${error_style}"Wine not available. Unable to determine exact place of the database."
        echo "Please install Wine first and ensure that the wine command is available."${default_style}
        exit 1
    fi
    # We get the value of the %APPDATA% environment variable.
    local wine_prefix_addpata_dir="$(LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine cmd /c "ECHO %APPDATA%" 2>/dev/null)"
    # And then we use winepath.exe to transform it to a unix path.
    local wine_prefix_appdata_dir="$(LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine winepath.exe "${wine_prefix_addpata_dir::-1}" 2>/dev/null)"
    local database_file="${wine_prefix_appdata_dir}/ADOxx/sqlitedbs/${database_name}.sqlite3"
    # Check again fi the database file exists.
    [ -f "${database_file}" ]
    local database_exists=$?
    if [ ${database_exists} -eq 0 ]
    then
        echo "Database file found: ${database_file}"
    fi

    # Approaches previously tried:

    # Using compgen to use all the user in the Wine prefix.
    # This also used an amount_of_database_files_found variable and a lot of
    # other code later on.
    #local wineprefix_users_folder="${wine_prefix_dir}/drive_c/users"
    #local database_file_in_user_folder="AppData/Roaming/ADOxx/sqlitedbs/${database_name}.sqlite3"
    #local found_database_files="$(compgen -G "${wineprefix_users_folder}/*/${database_file_in_user_folder}" 2>/dev/null)"

    # Using find to search for the database file in the Wine prefix folder.
    # This also used an amount_of_database_files_found variable and a lot of
    # other code later on.
    #local database_file_name="${database_name}.sqlite3"
    #local wineprefix_users_folder="${wine_prefix_dir}/drive_c/users"
    #local found_database_files="$(find "${wineprefix_users_folder}" -name "${database_file_name}" 2>/dev/null)"
fi

# If we still didn't find the database file then we assume it is missing.
if [ ${database_exists} -ne 0 ]
then
    echo "Database missing."
fi



# If the database already exists, then prompt the user on how to proceed.
if [ ${database_exists} -eq 0 ]
then
    echo ${warning_style}"Deleting the database file will also delete all models of the tool."${default_style}
    echo ${warning_style}"Make sure you no longer need them or that they have been exported as"${default_style}
    echo ${warning_style}".adl / .xml files first."${default_style}
    echo "Database file to delete: ${database_file}"
    prompt_user_default_no ${input_style}"Should the database be deleted? [y/N] or press CTRL-C to abort."${default_style}
    local delete_database=$?
fi



# Delete the database file if so desired.
if [ ${delete_database} -eq 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Deleting database."
    rm -f "${database_file}"
else
    echo "Database file is NOT being deleted."
fi



# Check if the database file has been removed and "return" 0 (true) or 1 (false).
if [ ${delete_database} -eq 0 ] && [ -f "${database_file}" ]
then
    echo ${error_style}"FAILED deleting database file."
    echo "Please delete the database file manually."${default_style}
    return 1
else
    return 0
fi
