#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# This file contains the uninstallation procedure for the tool. It uses many
# other script files to get the job done. It is provided as a separate file
# to allow the main uninstallation file seen by the user
# (uninstall_on_macos.sh) to perform updates to these installation scripts,
# e.g. downloading newer versions from the internet.

# The uninstallation script does not check after each step whether it has been
# successful. This is different from the installation script.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# tool_name - The name of the tool to be installed.
# tool_version - The version of the tool to be installed.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"

local grey=''
if [ ${no_style} -ne 0 ]
then
    local grey=$'\e[37m'
fi



# Print general information for the user.
echo
echo "-----------------------------------------------------------------   ${warning_style}*        ${grey}#"${default_style}
echo "|                                                                  ${warning_style}***     ${grey}###"${default_style}
echo "|         ADOxx UNinstaller for macOS                             ${warning_style}*****  ${grey}#####"${default_style}
echo "|         (Experimental Release, v1.0.0)                               ${grey}#######"${default_style}
echo "|                                                                    ${grey}#########"${default_style}
echo "-----------------------------------------------------------------  ${grey}###########"${default_style}
echo
echo "This script UNinstalls ADOxx or an ADOxx based tool on your 64-bit macOS."
echo ${warning_style}"Before starting the uninstallation, make sure that your system is up-to-date."${default_style}
echo "Provide --nostyle argument to the script to omit the use of styles and colors."
echo "Check the README.md and README-ADOxx.md for further details."
echo ${warning_style}"You can stop the execution at any point by pressing CTRL+C."${default_style}
echo "------------------------------------------------------------------------------"
echo
echo "Tool to be uninstalled: ${tool_name} ${tool_version}"
echo
echo "What is required:"
echo "Internet connection - some resources are downloaded from the internet."
echo "Administrative password - some steps can require administrative privileges."
echo "Manual steps - in some cases manual steps are needed."
echo "------------------------------------------------------------------------------"
read "?${input_style}Hit ENTER to continue"${default_style}
echo

echo "Components removed with this script:"
echo "1. Delete shortcuts."
echo "3. Delete tool directory."
echo "2. Delete database, including all models. (optional)"
echo "4. Delete Wine prefix. (optional)"
echo "5. Uninstall Wine. (optional)"
echo "6. Uninstall Homebrew. (optional)"
echo "------------------------------------------------------------------------------"
echo ${warning_style}"Pressing CTRL+C to abort the uninstallation."${default_style}
read "?${input_style}Hit ENTER to remove the above components."${default_style}
echo



# Check if user is root. Being root can mess up the uninstallation, so abort.
if [ $(id -un) = "root" ]
then
    echo ${error_style}"Do not execute this script as root (sudo)!"${default_style}
    exit
fi



# Perform the installation steps in their sequence.
echo "=== 1. Delete shortcuts."
"${0:a:h}/u1_delete_shortcuts.sh" "${@}"
echo

echo "=== 2. Delete tool directory."
"${0:a:h}/u2_delete_tool.sh" "${@}"
echo

echo "=== 3. Delete database, including all models. (optional)"
"${0:a:h}/u3_delete_database.sh" "${@}"
echo

echo "=== 4. Delete Wine prefix. (optional)"
"${0:a:h}/u4_delete_wineprefix.sh" "${@}"
echo

# When Homebrew is installed freshly on a system, then the brew command
# isn't available right away in the shell. Using `source` is only valid for
# this process and any sub-processes it spawns (e.g. .sh scripts executed).
# The parent process (i.e. terminal that executed this script) will most likely
# not have Homebrew available until it is reloaded.
# This is loaded BEFORE Wine, because Wine is managed through Homebrew.
if [ -e "~/.zprofile" ]
then
    source ~/.zprofile
fi

echo "=== 5. Uninstall Wine. (optional)"
"${0:a:h}/u5_uninstall_wine.sh" "${@}"
echo

echo "=== 6. Uninstall Homebrew. (optional)"
"${0:a:h}/u6_uninstall_homebrew.sh" "${@}"
echo

echo "=== Uninstallation script finished ==="
echo
