#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Copies the shortcuts for the tool.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# tool_name - The name of the tool to be installed.
# tool_version - The version of the tool to be installed.
# wine_prefix_dir - The Wine prefix directory.
# tool_installation_dir - The directory to install the tool to inside the Wine
#   prefix.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"

# Helper variables.
local full_tool_installation_dir="${wine_prefix_dir}/drive_c/${tool_installation_dir}"



# Check which of the shortcuts to copy are available.
# These have to follow the same logic as they are generated by the PDP.
if [ -n "${tool_modtk_shortcut_suffix}" ]
then
    local modtk_shortcut_name="${tool_name} ${tool_version} ${tool_modtk_shortcut_suffix}.app"
else
    local modtk_shortcut_name="${tool_name} ${tool_version}.app"
fi
[ -d "${0:a:h}/../${modtk_shortcut_name}" ]
local modtk_shortcut_provided=$?

local devtk_shortcut_name="${tool_name} ${tool_version} Development Toolkit.app"
[ -d "${0:a:h}/../${devtk_shortcut_name}" ]
local devtk_shortcut_provided=$?

local installation_dir_shortcut_name="${tool_name} ${tool_version} Installation Folder"



# Check if any of the shortcuts already exist.
if [ ${modtk_shortcut_provided} -eq 0 ] && [ -f "/Applications/${modtk_shortcut_name}" ]
then
    echo "${modtk_shortcut_name} already exists."
fi

if [ ${devtk_shortcut_provided} -eq 0 ] && [ -f "/Applications/${devtk_shortcut_name}" ]
then
    echo "${devtk_shortcut_name} already exists."
fi

if [ -L "/Applications/${installation_dir_shortcut_name}" ]
then
    echo "Link to ${installation_dir_shortcut_name} already exists."
fi



# Copy the available shortcuts.
# Whether to overwrite existing files or not is handled by the copy command.
if [ ${modtk_shortcut_provided} -eq 0 ]
then
    echo "Copying ${modtk_shortcut_name} shortcut"
    cp -R "${0:a:h}/../${modtk_shortcut_name}" "/Applications/"
    sudo xattr -r -d com.apple.quarantine "/Applications/${modtk_shortcut_name}"
    sudo chmod +x "/Applications/${modtk_shortcut_name}/Contents/MacOS/"*.sh
fi

if [ ${devtk_shortcut_provided} -eq 0 ]
then
    echo "Copying ${devtk_shortcut_name} shortcut"
    cp -R "${0:a:h}/../${devtk_shortcut_name}" "/Applications/"
    sudo xattr -r -d com.apple.quarantine "/Applications/${devtk_shortcut_name}"
    sudo chmod +x "/Applications/${devtk_shortcut_name}/Contents/MacOS/"*.sh
fi

# Updating the installation directory link.
if [ -L "/Applications/${installation_dir_shortcut_name}" ]
then
    echo "Removing existing link to installation directory: ${installation_dir_shortcut_name}"
    rm -r "/Applications/${installation_dir_shortcut_name}"
fi
echo "Creating link to installation directory: ${installation_dir_shortcut_name}"
ln -s "${full_tool_installation_dir}/" "/Applications/${installation_dir_shortcut_name}"



# Check if the shortcut files exist and "return" 0 (true) or 1 (false).
local shortcuts_missing=1

if [ ${modtk_shortcut_provided} -eq 0 ] && [ ! -d "/Applications/${modtk_shortcut_name}" ]
then
    echo "${modtk_shortcut_name} wasn't copied."
    local shortcuts_missing=0
fi

if [ ${devtk_shortcut_provided} -eq 0 ] && [ ! -d "/Applications/${modtk_shortcut_name}" ]
then
    echo "${devtk_shortcut_name} wasn't copied."
    local shortcuts_missing=0
fi

if [ ! -L "/Applications/${installation_dir_shortcut_name}" ]
then
    echo "Link to ${installation_dir_shortcut_name} wasn't created."
    local shortcuts_missing=0
fi

if [ ${shortcuts_missing} -eq 0 ]
then
    echo ${error_style}"FAILED copying (some) shortcuts."
    echo "Please copy and configure missing shortcuts manually."${default_style}
    return 1
else
    return 0
fi
