#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Check if the Wine prefix exists. If it does not exist or the --forceop
# argument is specified, then it tries to create the Wine prefix.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.
# wine_prefix_dir - The Wine prefix directory to create.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"



# Check if the Wine prefix already exists.
[ -d "${wine_prefix_dir}" ] &>/dev/null
local wine_prefix_exists=$?
if [ ${wine_prefix_exists} -eq 0 ]
then
    echo "Wine prefix already exists: ${wine_prefix_dir}"
else
    echo "Wine prefix missing: ${wine_prefix_dir}"
fi



# Create the Wine prefix by simply calling a command available in wine.
if [ ${wine_prefix_exists} -ne 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Trying to create Wine prefix: ${wine_prefix_dir}"
    # Note: using wincfg doesn't work. It throws an error: Bad CPU type in executable
    #LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all winecfg
    # wine64 is no longer installed with wine 10.x and using a symbolic link to point to wine64 to wine makes other things fail.
    #LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine64 cmd /c exit
    LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine cmd /c exit
fi



# Check if the Wine prefix exists and "return" 0 (true) or 1 (false).
if [ ! -d "${wine_prefix_dir}" ]
then
    echo ${error_style}"FAILED setting up Wine prefix."
    echo "Please set up the Wine prefix manually."${default_style}
    return 1
else
    return 0
fi
