#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Check if Homebrew is installed. If it is not installed or the --forceop
# argument is specified, then it tries to install Homebrew.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.

# Notes about some specifics of the script/s:
# * ${0:a:h} - The absolute path to the directory where THIS script is located.
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# Process passed arguments and load various relevant variables and functions.
source "${0:a:h}/common_script.sh"



# Check if Homebrew is installed.
command -v brew &>/dev/null
local homebrew_is_installed=$?
if [ ${homebrew_is_installed} -eq 0 ]
then
    local homebrew_version="$(echo "$(brew --version)")"
    echo "${homebrew_version} already installed."
else
    # Homebrew can be installed either /opt/homebrew (Apple Silicon) or
    # /usr/local (macOS Intel).
    if [ -e "/opt/homebrew/bin/brew" ]
    then
        echo "It appears that homebrew is installed in /opt/homebrew/bin/brew"
        echo "Please run the following command and resolve any issues before trying the"
        echo "installation again: /opt/homebrew/bin/brew doctor"
        return 1
    elif [ -e "/usr/local/bin/brew" ]
    then
        echo "It appears that homebrew is installed in /usr/local/bin/brew"
        echo "Please run the following command and resolve any issues before trying the"
        echo "installation again: /usr/local/bin/brew doctor"
        return 1
    else
        echo "Homebrew does not seem to be installed."
    fi
fi



# Update Homebrew if desired by the user.
if [ ${homebrew_is_installed} -eq 0 ]
then
    local update_homebrew=1
    if [ ${force_operation} -eq 0 ]
    then
        local update_homebrew=0
    else
        prompt_user_default_yes ${input_style}"Should Homebrew and its formulae be updated and upgraded? [Y/n] or press CTRL-C to abort."${default_style}
        local update_homebrew=$?
    fi
    if [ ${update_homebrew} -eq 0 ]
    then
        echo "Updating Homebrew."
        brew doctor
        brew update
        brew upgrade
    fi
fi



# Install Homebrew.
if [ ${homebrew_is_installed} -ne 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Installing Homebrew"
    /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"

    # When Homebrew is installed freshly on a system, then the brew command
    # isn't available right away in the shell. The solution below is only valid
    # for this process and any sub-process it spawns, but not for scripts
    # (parent processes) that call this script.
    # ANY SCRIPT THAT CALLS THIS MUST ALSO UPDATE THEIR SHELL SESSION IN SUCH
    # A CASE! Easiest way to do that seems to be to run: source ~/.zprofile
    # If the brew command is not yet available, but it does exist, then adapt
    # the local shell environment.
    command -v brew &>/dev/null
    if [ $? -ne 0 ]
    then
        if [ -e "/opt/homebrew/bin/brew" ]
        then
            eval "$(/opt/homebrew/bin/brew shellenv)"
            if [ ! -e "~/.zprofile" ] || [ -z $(grep 'eval "$(/opt/homebrew/bin/brew shellenv)"' "~/.zprofile") ]
            then
                echo "Adding brew to ~/.zprofile"
                (echo; echo 'eval "$(/opt/homebrew/bin/brew shellenv)"') >> ~/.zprofile
            fi
        elif [ -e "/usr/local/bin/brew" ]
        then
            eval "$(/usr/local/bin/brew shellenv)"
            if [ ! -e "~/.zprofile" ] || [ -z $(grep 'eval "$(/usr/local/bin/brew shellenv)"' "~/.zprofile") ]
            then
                echo "Adding brew to ~/.zprofile"
                (echo; echo 'eval "$(/usr/local/bin/brew shellenv)"') >> ~/.zprofile
            fi
        fi
    fi

    # Running doctor "just in case"
    brew doctor
    # Hopefully brew update isn't necessary after installing homebrew...
fi



# Check if Homebrew is installed and "return" 0 (true) or 1 (false).
# Note: the parent process MUST update their shell to get to use brew.
command -v brew &>/dev/null
local homebrew_is_installed=$?
if [ ${homebrew_is_installed} -ne 0 ]
then
    echo ${error_style}"FAILED installing Homebrew."
    echo "Please install Homebrew manually."${default_style}
    return 1
else
    return 0
fi
