#!/bin/zsh

# Version: 1.0.0
# Date: 2023-10-04
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Processes arguments and sets various variables based on those arguments. It
# also provides several variables, like loading variables that hold the tool's
# details (tool_config.env). To use this script simply call it from a
# parent script with the `source` command.

# It also contains various useful functions that are used in other scripts.

# The basic arguments are:
# --nostyle - Sets nostyle variable to 0 (true). Default is 1 (false).
# --forceop - Sets force_operation variable to 0 (true). Default is 1 (false).

# It also has a special argument, which allows to set any variable to a value:
# --d-VARIABLE=VALUE - Sets the value of the VARIABLE to VALUE. For example
#   --d-database=adoxx18 will set the variable database to the value adoxx18.
#   If VALUE contains spaces they have to be escaped (using \ ) or just put the
#   VALUE in double quotes, like --d-somevar="value with spaces"



# Simple function that prompts the user a "yes or no" question, with defaulting
# to "no". The "return value" $? is 0 for yes or non-0 for no.
function prompt_user_default_no {
    local tempval=""
    if [ $# -gt 0 ]
    then
        read tempval"?${*} "
    else
        read tempval"?Yes or no? (default: No) "
    fi
    tempval=$(echo ${tempval} | tr '[:upper:]' '[:lower:]')
    [ -n ${tempval} ] && [[ ${tempval} = y* ]]
}



# Simple function that prompts the user a "yes or no" question, with defaulting
# to "yes". The "return value" $? is 0 for yes or non-0 for no.
function prompt_user_default_yes {
    local tempval=""
    if [ $# -gt 0 ]
    then
        read tempval"?${*} "
    else
        read tempval"?Yes or no? (default: Yes) "
    fi
    tempval=$(echo ${tempval} | tr '[:upper:]' '[:lower:]')
    [ -z ${tempval} ] || [[ ${tempval} != n* ]]
}



# Load the tool's details as variables so they can be used in the scripts.
if [ -f "${0:a:h}/../tool_config.env" ]
then
    source "${0:a:h}/../tool_config.env"
fi

# Variables that can be set via arguments.
local no_style=1
local force_operation=1

# Variables for styling output.
    local error_style=''     # Used for errors.
    local input_style=''     # Used for messages that should get attention.
    local warning_style=''   # Used to indicate user input.
    local default_style=''   # Default style of text.
# colors not used:
#    local green_style=$'\e[1;32m'
#    local magenta_style=$'\e[1;35m'
#    local cyan_style=$'\e[1;36m'



# Process passed arguments and set variables accordingly.
# Performed after loading tool_config.env to allow overwriting them.
for argvar in ${@}
do
    if [ "${argvar}" = "--nostyle" ]
    then
        local no_style=1
    elif [ "${argvar}" = "--forceop" ]
    then
        local force_operation=0
    elif [ "${argvar[1,4]}" = "--d-" ] && [[ "${argvar}" =~ "=" ]]
    then
        # This allows to set other variables in the script passed as arguments.
        eval local $(echo ${argvar[5,-1]} | cut -d'=' -f 1)=\"$(echo ${argvar[5,-1]} | cut -d'=' -f 2-)\"
    fi
done



# Update the style variables according to the state of no_style.
if [ ${no_style} -ne 0 ]
then
    local error_style=$'\e[1;31m'
    local warning_style=$'\e[33m'
    local input_style=$'\e[1;34m'
    local default_style=$'\e[0m'
fi
