## tourlquerystring

"Unpack" the contents of a map into a URL query string format.



### Syntax

```AdoScript
tourlquerystring ( amap:mapValue [ , allowed:strValue ] )
# --> RETURNS strValue
```



### Parameters

* `amap` (mapValue) - The map with the content that should be "unpacked" into a
  query string.
* `allowed` (strValue, optional) - Contains all characters that are allowed to
  appear normally (not encoded) in the result.



### Returns

* (strValue) - The specified content, where all characters not found in
  `allowed` are URI encoded (`%xx`, "percent-encoding").



### Remarks

This function makes it easier to build URL query strings by transforming a
structured value, specifically a map. The map is considered to be shallow, i.e.
any of its other structured elements are treated as their string
representation. Characters are encoded / escaped using typical URI encoding
(`%xx`), based on the value of `allowed`.

If `allowed` is not specified, then the set of allowed characters is the same
as by the `encodeURIComponent()` function from JavaScript, specifically the
characters according to the [MDN web docs](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURIComponent).

Be aware of where the result is being used. If it is for example used in
`encodeuri()`, then the query string will be encoded twice!



### Examples

Simple input map with one key-value pair. Result is encoded using default
allowed characters, similar to JavaScript's `encodeURIComponent()` function.

```AdoScript
tourlquerystring({"key": "value"})
# Result: key=value
```

Using a map with several key-value pairs, not all of which are strings. The
result is encoded using default allowed characters, similar to JavaScript's
`encodeURIComponent()` function.

```AdoScript
tourlquerystring({"key": "value", 42: "milk&soy", "one": "mo/re#wi=th"})
# Result: key=value&one=mo%2Fre%23wi%3Dth&42=milk%26soy
```

Simple input map with one key-value pair. Result has encoded everything that
is not a vowel.

```AdoScript
tourlquerystring({"key": "value"}, "AEIOUaeiou")
# Result: %6Be%79=%76a%6Cue
```

Using a map with several key-value pairs, not all of which are strings. The
result has encoded everything that is not a vowel.

```AdoScript
tourlquerystring({"key": "value", 42: "milk&soy", "one": "mo/re#wi=th"}, "AEIOUaeiou")
# Result: %6Be%79=%76a%6Cue&o%6Ee=%6Do%2F%72e%23%77i%3D%74%68&%34%32=%6Di%6C%6B%26%73o%79
```

Using a map with several key-value pairs, not all of which are strings. The 2nd
parameter specifies explicitly the allowed characters to be the same as used by
JavaScript's `encodeURIComponent()` function.

```AdoScript
tourlquerystring({"key": "value", 42: "milk&soy", "one": "mo/re#wi=th"},
  "!'()*-.0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~")
# Result: key=value&one=mo%2Fre%23wi%3Dth&42=milk%26soy
```

Using a map with several key-value pairs, not all of which are strings. The 2nd
parameter specifies explicitly the allowed characters to be the same as used by
JavaScript's `encodeURI()` function. This will not encode characters (&, =)
that have a special meaning in the rest of the URL.

```AdoScript
tourlquerystring({"key": "value", 42: "milk&soy", "one": "mo/re#wi=th"},
  "!#$&'()*+,-./0123456789:;=?@ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~")
# Result: key=value&one=mo/re#wi=th&42=milk&soy
```

Examples for some strings that can be used for `allowed`.

```AdoScript
# Characters allowed by JS encodeURIComponent(). It should be safe to encode
# specific parts of a URL.
"!'()*-.0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~"

# Characters allowed by JS encodeURI(). It should be safe to use for encoding
# an entire URL.
"!#$&'()*+,-./0123456789:;=?@ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~"

# Based on several sources from the internet, including the URI specification.
"!\"$'()*,-./0123456789;ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~"

# All except non-ASCII characters, ASCII control characters (0x00 to 0x20 and
# 0x7F), # (0x23), % (0x25), & (0x26), + (0x2B), = (0x3D) and ? (0x3F)
"!\"$'()*,-./0123456789:;<>@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~"

# All except non-ASCII characters, ASCII control characters (0x00 to 0x20 and
# 0x7F), % (0x25), & (0x26) and = (0x3D)
"!\"#$'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~"
```



### See Also

* `encodeuri()`



### Versions and Changes

Available since ADOxx 1.8.0
