## SET_ATTR_VAL

Sets the value for an attribute of one object / instance.



### Syntax

```AdoScript
CC "Core" SET_ATTR_VAL objid:idValue ( attrid:idValue | attrname:strValue )
    val:anyValue [ as-string:boolValue ] [ check-domain:boolValue ]
    [ create-save-action:boolValue ]
# --> RESULT ecode:intValue
```



### Parameters

* `objid` (idValue) - The ID of the object where the value should be changed.
* `attrid` (idValue) - The ID of the attribute whose value should be changed.
* `attrname` (strValue) - The name of the attribute whose value should be
  changed.
* `val` (anyValue) - The new value to set the attribute to.
* `as-string` (boolValue, optional) - When true, then the provided `val` must
  be a string and is converted to the attribute's type before assigning it. The
  default is false.
* `check-domain` (boolValue, optional) - When true, then changing an INTERREF
  type attribute will check if the value fits with the specified domain for the
  attribute. The default is false.
* `create-save-action` (boolValue, optional) - When true, then the Modelling
  Toolkit can save changes to the attributes of a library. The default is
  false.



### Returns

* `ecode` (intValue) - Contains the error code or 0 in case of success.



### Remarks

Either `attrid` or `attrname` have to be provided as the parameter.

This command is mainly used to modify the content of instances on "model"
level, but it can also be used to modify content on "meta-model" level. This is
possible, because the object / instance provided by ID can be a modeling
object, a connector, a record row, an attribute profile, a library, a
(modeling) class, a relation class, a model etc. See [Examples](#examples)
section for how it all links together.

When `as-string` is specified, then `val` must be a string which is converted
to the attribute type (e.g. INTEGER) internally. Otherwise the type of val must
match the type of the attribute.

Modifying an attribute value will only work if the model / library is loaded at
the moment when the command is called. Furthermore the change will only be
persisted, if the corresponding "owner" (model or library) is saved afterwards
using for example `"Core" SAVE_MODEL` for the contents in a model or
`"Core" SAVE_LIBRARY` for the contents of an attribute profile or a library.

The `check-domain` parameter is only relevant for attributes of type INTERREF.
The check is performed before the value of the attribute is set. The command
aborts with an error and the attribute value is not changed when the domain
check fails, like when trying to reference an invalid target.

The `create-save-action` parameter must be provided (true) when modifying
parts on library level (library attributes, attribute default values etc.) to
be able to save them. Otherwise the changes are not persisted beyond the
session.

Some attribute's default values can not be changed, e.g. the "GraphRep" or
"AttrRep" of most of the predefined meta-model classes (`__D_event__`,
`__D_container__`, `__D_end__` etc.)



### Examples

Change the value of the "Description" attribute of the first object from the
currently selected list. This example assumes that the model is already loaded.
Furthermore, it does not save the changes.

```
# Get the ID of the first object from the currently selected list.
CC "Modeling" GET_SELECTED
SETL idFirstSelected:(VAL token(objids, 0))
# Change the object's "Description" attribute value to "Hello World!".
CC "Core" SET_ATTR_VAL objid:(idFirstSelected) attrname:("Description")
    val:("Hello World!")
# Use either "Core" SAVE_MODEL to save the changes or "Modeling" SET_MODIFIED
# to indicate to the user that changes have been performed.
```

Set the name of the first object from the currently selected list. This example
assumes that the model is already loaded. Furthermore, it does not save the
changes.

```AdoScript
# Get all selected objects and extract from the list the first objectid.
CC "Modeling" GET_SELECTED
IF (objids = "") {
  CC "AdoScript" ERRORBOX "No object has been selected!"
  EXIT
}
SETL idSelected:(VAL token(objids, 0))

# Get the class of the selected object.
CC "Core" GET_CLASS_ID objid:(idSelected)
SETL idSelectedClass:(classid)

# Get the attribute "Name" of the class.
CC "Core" GET_ATTR_ID classid:(idSelectedClass) attrname:"Name"
IF (ecode) {
  CC "AdoScript" ERRORBOX "The selected object does not contain an attribute called \"Name\"!"
  EXIT
}
SETL idNameAttr:(attrid)

# Set the name of the selected object.
CC "Core" SET_ATTR_VAL objid:(idSelected) attrid:(idNameAttr) val:"Seppl"
IF (ecode) {
  CC "AdoScript" ERRORBOX "Could not set the attribute value!"
  EXIT
}

# Use either "Core" SAVE_MODEL to save the changes or "Modeling" SET_MODIFIED
# to indicate to the user that changes have been performed.
```

Change the "Description" attribute value of the currently active dynamic model.

```AdoScript
# Get the current active model.
CC "Modeling" GET_ACT_MODEL
IF (modelid < 0) {
  CC "AdoScript" ERRORBOX "Open a dynamic model first!"
  EXIT
}
SETL idActiveModel:(modelid)

# Get the model attribute "Description".
CC "Core" GET_ATTR_ID classid:bp-model attrname:"Description"
IF (ecode) {
  CC "AdoScript" ERRORBOX "Model does not contain an attribute called \"Description\"!"
  EXIT
}
SETL idDescriptionAttr:(attrid)

# Set the model attribute.
CC "Core" SET_ATTR_VAL objid:(idActiveModel) attrid:(idDescriptionAttr)
    val:"Model description here!"
IF (ecode) {
  CC "AdoScript" ERRORBOX "Could not set the attribute value!"
  EXIT
}

# Use either "Core" SAVE_MODEL to save the changes or "Modeling" SET_MODIFIED
# to indicate to the user that changes have been performed.
```

Change the default value of the "TestClass" class's "GraphRep" attribute.

```AdoScript
# Get the IDs of the currently loaded library.
CC "Core" GET_CURRENT_LIBRARIES
SETL idCurBpLib:(bplibid)
# Get the class ID of the class that "owns" the attribute.
CC "Core" GET_CLASS_ID classname:("TestClass")
SETL idClass:(classid)
# Change the default value of the "GraphRep" attribute.
CC "Core" SET_ATTR_VAL objid:(idClass) attrname:("GraphRep") create-save-action
    val:("GRAPHREP\n" +
        "SHADOW off\n\n" +
        "PEN style:solid\n" +
        "FILL color:(\"blue\")\n" +
        "RECTANGLE x:-1cm y:-0.5cm w:2cm h:1cm")
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
```

Change the default value of the "Description" model attribute.

```AdoScript
# Get the class of the selected object.
CC "Core" GET_CLASS_ID classname:"__ModelTypeMetaData__" bp-library
SETL idMTClass:(classid)

# Get the attribute "Description" of the class.
CC "Core" GET_ATTR_ID classid:(idMTClass) attrname:"Description"
IF (ecode) {
  CC "AdoScript" ERRORBOX "The selected class does not contain an attribute called \"Description\"!"
  EXIT
}
SETL idMTDescription:(attrid)

# Set the default value for the "Description" attribute.
CC "Core" SET_ATTR_VAL objid:(idMTClass) attrid:(idMTDescription)
    create-save-action val:("Model description here!")
IF (ecode) {
  CC "AdoScript" ERRORBOX "Could not set the attribute value!"
  EXIT
}

# Use "Core" SAVE_LIBRARY to save the changes.
```

Modify the dynamic library's "Modi" attribute by adding a new model type
definition.

```AdoScript
# Get the IDs of the currently loaded library.
CC "Core" GET_CURRENT_LIBRARIES
SETL idCurBpLib:(bplibid)
# Get the class ID of the library that "owns" the "Modi" attribute.
CC "Core" GET_CLASS_ID objid:(idCurBpLib)
SETL idCurBpLibClass:(classid)
# Get the attribute ID of "Modi"
CC "Core" GET_ATTR_ID classid:(idCurBpLibClass) attrname:("Modi")
SETL idModi:(attrid)
# Get the current value of "Modi", so that we can append to it.
CC "Core" GET_ATTR_VAL objid:(idCurBpLib) attrid:(idModi)
SETL sOldModiVal:(val)
# Append a "New Model Type" to the "Modi" attribute.
CC "Core" SET_ATTR_VAL objid:(idCurBpLib) attrid:(idModi) create-save-action
    val:(sOldModiVal + "\n\n"+
        "MODELTYPE \"New Model Type\" from:none\n")
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
```



### See Also

* `"Core" GET_ATTR_VAL`
* `"Core" GET_FACET_VAL`
* `"Core" SET_FACET_VAL`
* `"Core" LOAD_MODEL`
* `"Core" SAVE_MODEL`
* `"Core" LOAD_LIBRARY`
* `"Core" SAVE_LIBRARY`
* `"Modeling" SET_MODIFIED`



### Versions and Changes

Available since ADOxx 1.3

*ADOxx 1.8.0*
* The command now returns an error when trying to change the attribute of a
  model that is not loaded.
