## BeforeRemoveObject

This event is triggered by the model editor before an object is to be deleted
from a model. An object here is a modelling instance or connector.



### Syntax

```AdoScript
ON_EVENT "BeforeRemoveObject" {
  # PARAMETERS: instid:idValue classid:idValue modelid:idValue
  # EXIT: -3, -2, -1, 0, >0
}
```



### Parameters

* `instid` (idValue) - The ID of the object.
* `classid` (idValue) - The ID of the class the object belongs to.
* `modelid` (idValue) - The ID of the model where the object is located.



### Exit values

Controls whether the specific object is deleted or not.

* `0`  - Normal behavior.
* `-1` - Abort without error.
* `-2` - Abort with error. Aborts the entire delete operation as a whole.
* `-3` - Delete the object, but skip the check for referencing objects.
* `>0` - Abort with core error code. Aborts the entire delete operation as a
  whole.



### Remarks

That event is sent before any delete related actions are performed. Such
actions are, e.g the check if an object is still referenced by another object,
etc.

When multiple object's are deleted at once, then the event is sent for each
single object and a different reaction can be returned for each one of them.
This allows for an adaptation of the objects to be deleted.

The behavior concerning a selection of several objects is as follows:
1. If for at least one of the objects the event exits with an error (>0 or -2),
  the whole delete action is cancelled.
2. If for some of the objects -1 is returned and for some 0, then the selection
  is adjusted accordingly and the usual deletion behavior proceeds.

When -3 is returned for some objects, those objects will be deleted, but the
check if these objects are still referenced by other objects will not be
performed.

Note that this event is only triggered by issuing an explicit delete action by
the user. It is not triggered in the following cases:
* An internal delete when cutting objects.
* Using the `CC "Core" DELETE_OBJ` or `CC "Core" DELETE_OBJS` command.
* ...



### Examples

Show a popup with the name and class name of each object that is being deleted.

```AdoScript
ON_EVENT "BeforeRemoveObject" {
  SETL idDeleteObj:(instid)
  # Get the name of the object that is being copied.
  CC "Core" GET_OBJ_NAME objid:(idDeleteObj)
  IF (ecode) { EXIT 0 }
  SETL sDeleteObjectName:(objname)
  # Get the class name of the object that is being copied.
  CC "Core" GET_CLASS_ID objid:(idDeleteObj)
  SETL idDeleteClass:(classid)
  CC "Core" GET_CLASS_NAME classid:(idDeleteClass)
  SETL sDeleteClassName:(classname)
  # Show both names.
  CC "AdoScript" INFOBOX (sDeleteObjectName + "\n" + sDeleteClassName)
}
```

Don't delete objects whose name contains "directors cut".

```AdoScript
ON_EVENT "BeforeRemoveObject" {
  SETL idDeleteObj:(instid)
  # Get the name of the object that is being copied.
  CC "Core" GET_OBJ_NAME objid:(idDeleteObj)
  IF (ecode) {
    # Allow deleting object's whose name we can't find out.
    EXIT 0
  }
  SETL sDeleteObjectName:(objname)
  # Abort deleting the object if it's name contains "directors cut".
  IF (search(sDeleteObjectName, "directors cut", 0) >= 0) {
    EXIT -1
  }
}
```

Abort the entire delete operation when one of the object's name contains
"directors cut".

```AdoScript
ON_EVENT "BeforeRemoveObject" {
  SETL idDeleteObj:(instid)
  # Get the name of the object that is being copied.
  CC "Core" GET_OBJ_NAME objid:(idDeleteObj)
  IF (ecode) {
    # Allow deleting object's whose name we can't find out.
    EXIT 0
  }
  SETL sDeleteObjectName:(objname)
  # Abort deleting the object if it's name contains "directors cut".
  IF (search(sDeleteObjectName, "directors cut", 0) >= 0) {
    EXIT -2
  }
}
```



### See Also

* `"Modeling" CUT_SELECTED`
* `ON_EVENT "BeforeDeleteInstance"`
