/*
//  Create ADOxx database objects
//  ------------------------------
//
//  Description:
//      Creates tables, indexes and triggers of an ADOxx database.
//      The script provides the prerequisites for the tool
//      'ADOxx database creation' to be used with the option
//      'Create tables' deactivated.
//
//  Prerequisites:
//    - Open connection of administrator user (DBA).
//    - User 'ADOxx' exists with password 'BPMS' and owns the role 'CONNECTION'.
//    - ADOxx owns the role 'RESOURCE', or 'UNLIMITED QUOTA' on its 
//      default tablespace and temporary tablespace, respectively.
//  --------------------------------
//  Version: 7.0
//  Created: 07-12-2000
//  Last change: 18-08-2016
//
//  (c) BOC GmbH, Vienna(A) 2001 - 2016
*/ 




REM ********** Create tables **********************

CREATE TABLE ADOxx.class(
  id NUMBER(12) NOT NULL,
  name VARCHAR(250),
  superclass NUMBER(12),
  modclassid NUMBER(12),
  modclasstype NUMBER(12),
  connrelid NUMBER(12),
  connreltype NUMBER(12),
  compid NUMBER(12));

CREATE TABLE ADOxx.attribute(
  id NUMBER(12) NOT NULL,
  inst_attr NUMBER(6),
  name VARCHAR(255),
  owner NUMBER(12) NOT NULL,
  type NUMBER(6),
  libid NUMBER(12));

CREATE TABLE ADOxx.instance(
  id NUMBER(12) NOT NULL,
  modelid NUMBER(12),
  class NUMBER(12) NOT NULL,
  name VARCHAR(250),
  modinstid NUMBER(12),
  modinsttype NUMBER(12),
  connrelid NUMBER(12),
  connreltype NUMBER(12));

CREATE TABLE ADOxx.facet_double(
  attribute NUMBER(12) NOT NULL,
  id NUMBER(12) NOT NULL,
  name VARCHAR(255),
  type NUMBER(6),
  value NUMBER(38,6));

CREATE TABLE ADOxx.facet_integer(
  attribute NUMBER(12) NOT NULL,
  id NUMBER(12) NOT NULL,
  name VARCHAR(255),
  type NUMBER(6),
  value NUMBER(12));

CREATE TABLE ADOxx.facet_string(
  attribute NUMBER(12) NOT NULL,
  id NUMBER(12) NOT NULL,
  name VARCHAR(255),
  type NUMBER(6),
  value VARCHAR(2000),
  value2 VARCHAR(1700)) ;

CREATE TABLE ADOxx.instattr_double(
  attribute NUMBER(12) NOT NULL,
  instance NUMBER(12) NOT NULL,
  value NUMBER(38,6));

CREATE TABLE ADOxx.instattr_integer(
  attribute NUMBER(12) NOT NULL,
  instance NUMBER(12) NOT NULL,
  value NUMBER(12));

CREATE TABLE ADOxx.instattr_string(
  attribute NUMBER(12) NOT NULL,
  instance NUMBER(12) NOT NULL,
  value VARCHAR(2000),
  value2 VARCHAR(1700));

CREATE TABLE ADOxx.instattr_long(
  attribute NUMBER(12) NOT NULL,
  instance NUMBER(12) NOT NULL,
  value LONG);

CREATE TABLE ADOxx.instattr_record(
  attribute NUMBER(12) NOT NULL,
  instance NUMBER(12) NOT NULL,
  recid NUMBER(12) NOT NULL,
  preid NUMBER(12) NOT NULL,
  modrecid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.reltype(
  id NUMBER(12) NOT NULL,
  afrom NUMBER(12) NOT NULL,
  name VARCHAR(255),
  ato NUMBER(12) NOT NULL,
  libid NUMBER(12) NOT NULL,
  type NUMBER(6));

CREATE TABLE ADOxx.relinstance(
  id NUMBER(12) NOT NULL,
  afrom NUMBER(12) NOT NULL,
  reltype NUMBER(12) NOT NULL,
  ato NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.adouser(
  userid NUMBER(12) NOT NULL,
  name VARCHAR(50) NOT NULL,
  password VARCHAR(50),
  smisettings VARCHAR(2000),
  smisettings1 VARCHAR(1701),
  mocsettings VARCHAR(2000),
  mocsettings1 VARCHAR(1701),
  gdtsettings LONG,
  usercomment VARCHAR(255),
  cussettings VARCHAR(2000),
  cussettings1 VARCHAR(1701));

CREATE TABLE ADOxx.userpref(
  userid NUMBER(12) NOT NULL,
  userprefs LONG);

CREATE TABLE ADOxx.permissions(
  userid NUMBER(12) NOT NULL,
  objectid NUMBER(12) NOT NULL,
  permission NUMBER(6) NOT NULL);

CREATE TABLE ADOxx.adolock(
  userid NUMBER(12) NOT NULL,
  objectid NUMBER(12) NOT NULL,
  locktype NUMBER(6) NOT NULL,
  locktime DATE);

CREATE TABLE ADOxx.adoid(
  lastid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.modelthread(
  modelid NUMBER(12) NOT NULL,
  name VARCHAR(250),
  type VARCHAR(250),
  libid NUMBER(12) NOT NULL,
  classid NUMBER(12),
  rellibinstid NUMBER(12),
  rellibclassid NUMBER(12));

CREATE TABLE ADOxx.modelversion(
  id NUMBER(12) NOT NULL,
  threadid NUMBER(12) NOT NULL,
  version VARCHAR(30));

CREATE TABLE ADOxx.modelgroup(
  groupid NUMBER(12) NOT NULL,
  name VARCHAR(152),
  supergroup NUMBER(12),
  connrelid NUMBER(12) NOT NULL,
  libid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.modeltogroup(
  refid NUMBER(12) NOT NULL,
  modelid NUMBER(12) NOT NULL,
  groupid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.amodel(
  amodelid NUMBER(12) NOT NULL,
  name VARCHAR(255),
  rellibinstid NUMBER(12),
  rellibclassid NUMBER(12),
  libid NUMBER(12));

CREATE TABLE ADOxx.modeltoamodel(
  modelid NUMBER(12) NOT NULL,
  amodelid NUMBER(12) NOT NULL,
  conrelninst NUMBER(12),
  conrelntype NUMBER(12));

CREATE TABLE ADOxx.usergroup(
  groupid NUMBER(12) NOT NULL,
  name VARCHAR(152) NOT NULL,
  perms VARCHAR(2000));

CREATE TABLE ADOxx.userstogroups(
  refid NUMBER(12) NOT NULL,
  userid NUMBER(12) NOT NULL,
  groupid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.modelgroupperms(
  permid NUMBER(12) NOT NULL,
  ugroupid NUMBER(12) NOT NULL,
  mgroupid NUMBER(12) NOT NULL,
  permtype NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.adotransact(
  dblock NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.dbinfo(
  name VARCHAR(30) NOT NULL,
  value VARCHAR(250) NOT NULL);

CREATE TABLE ADOxx.interref(
  instance NUMBER(12) NOT NULL,
  attribute NUMBER(12) NOT NULL,
  tomtype VARCHAR(250) NOT NULL,
  tomname VARCHAR(250) NOT NULL,
  tomversion VARCHAR(30),
  tocname VARCHAR(250),
  toiname VARCHAR(250),
  libid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.repodir(
  id NUMBER(12) NOT NULL,
  name VARCHAR(230) NOT NULL,
  parent NUMBER(12) NOT NULL,
  libid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.repoinstance(
  id NUMBER(12) NOT NULL,
  name VARCHAR(200) NOT NULL,
  class NUMBER(12) NOT NULL,
  dir NUMBER(12) NOT NULL,
  thread NUMBER(12) NOT NULL,
  version VARCHAR(30),
  libid NUMBER(12) NOT NULL,
  modid NUMBER(12) NOT NULL,
  attr NUMBER(12));

CREATE TABLE ADOxx.instattr_reporef(
  instance NUMBER(12) NOT NULL,
  attribute NUMBER(12) NOT NULL,
  topname VARCHAR(251) NOT NULL,
  libid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.message(
  id NUMBER(12) NOT NULL,
  recvid NUMBER(12) NOT NULL,
  stime DATE NOT NULL,
  sender VARCHAR(50) NOT NULL,
  subject VARCHAR(255),
  message LONG VARCHAR,
  state CHAR(1));

CREATE TABLE ADOxx.filestore(
  name VARCHAR(250) NOT NULL,
  data LONG RAW,
  info VARCHAR(255));

CREATE TABLE ADOxx.rdmmattribute(
  id NUMBER(12) NOT NULL,
  inst_attr NUMBER(6),
  name VARCHAR(255),
  owner NUMBER(12) NOT NULL,
  type NUMBER(6),
  libid NUMBER(12) NOT NULL);

CREATE TABLE ADOxx.rdmmfacet(
  id NUMBER(12) NOT NULL,
  attribute NUMBER(12) NOT NULL,
  name VARCHAR(255),
  type NUMBER(6),
  intvalue NUMBER(12),
  realvalue NUMBER(38,6),
  strvalue VARCHAR(2000),
  strvalue2 VARCHAR(1700),
  libid NUMBER(12) NOT NULL);

REM ******* Define constraints ***********

ALTER TABLE ADOxx.class ADD PRIMARY KEY(id);

ALTER TABLE ADOxx.attribute ADD PRIMARY KEY (id,owner);

ALTER TABLE ADOxx.instance ADD PRIMARY KEY(id);

ALTER TABLE ADOxx.facet_double ADD PRIMARY KEY (attribute,id);

ALTER TABLE ADOxx.facet_integer ADD PRIMARY KEY (attribute,id);

ALTER TABLE ADOxx.facet_string ADD PRIMARY KEY (attribute,id);

ALTER TABLE ADOxx.instattr_double ADD PRIMARY KEY (instance,attribute);

ALTER TABLE ADOxx.instattr_integer ADD PRIMARY KEY (instance,attribute);

ALTER TABLE ADOxx.instattr_string ADD PRIMARY KEY (instance,attribute);

ALTER TABLE ADOxx.instattr_long ADD PRIMARY KEY (instance,attribute);

ALTER TABLE ADOxx.instattr_record ADD PRIMARY KEY(recid);

ALTER TABLE ADOxx.reltype ADD PRIMARY KEY (id);

ALTER TABLE ADOxx.relinstance ADD PRIMARY KEY (id);

ALTER TABLE ADOxx.adouser ADD PRIMARY KEY (userid);

ALTER TABLE ADOxx.userpref ADD PRIMARY KEY (userid);

ALTER TABLE ADOxx.permissions ADD PRIMARY KEY (userid,objectid,permission);

ALTER TABLE ADOxx.adolock ADD PRIMARY KEY (userid,objectid,locktype);

ALTER TABLE ADOxx.modelthread ADD PRIMARY KEY (modelid);

ALTER TABLE ADOxx.modelversion ADD PRIMARY KEY (id);

ALTER TABLE ADOxx.modelversion ADD CONSTRAINT exVThread FOREIGN KEY(threadid) REFERENCES ADOxx.modelthread(modelid);

ALTER TABLE ADOxx.modelgroup ADD PRIMARY KEY (groupid);

ALTER TABLE ADOxx.modelgroup ADD CONSTRAINT mg_existSupGroup FOREIGN KEY(supergroup) REFERENCES ADOxx.modelgroup(groupid);

ALTER TABLE ADOxx.modeltogroup ADD PRIMARY KEY (refid);

ALTER TABLE ADOxx.modeltogroup ADD CONSTRAINT mtg_existMGroup FOREIGN KEY(groupid) REFERENCES ADOxx.modelgroup(groupid);

ALTER TABLE ADOxx.modeltogroup ADD CONSTRAINT mtg_existModel FOREIGN KEY(modelid) REFERENCES ADOxx.modelthread(modelid);

ALTER TABLE ADOxx.amodel ADD PRIMARY KEY (amodelid);

ALTER TABLE ADOxx.modeltoamodel ADD PRIMARY KEY (modelid,amodelid);

ALTER TABLE ADOxx.modeltoamodel ADD CONSTRAINT mtam_existAModel FOREIGN KEY(amodelid) REFERENCES ADOxx.amodel(amodelid);

ALTER TABLE ADOxx.usergroup ADD PRIMARY KEY (groupid);

ALTER TABLE ADOxx.userstogroups ADD PRIMARY KEY (refid);

ALTER TABLE ADOxx.userstogroups ADD CONSTRAINT utg_existUser FOREIGN KEY(userid) REFERENCES ADOxx.adouser(userid);

ALTER TABLE ADOxx.userstogroups ADD CONSTRAINT utg_existUGroup FOREIGN KEY(groupid) REFERENCES ADOxx.usergroup(groupid);

ALTER TABLE ADOxx.modelgroupperms ADD PRIMARY KEY (permid);

ALTER TABLE ADOxx.modelgroupperms ADD CONSTRAINT mgp_existUGroup FOREIGN KEY(ugroupid) REFERENCES ADOxx.usergroup(groupid);

ALTER TABLE ADOxx.modelgroupperms ADD CONSTRAINT mgp_existMGroup FOREIGN KEY(mgroupid) REFERENCES ADOxx.modelgroup(groupid);

ALTER TABLE ADOxx.dbinfo ADD PRIMARY KEY (name);

ALTER TABLE ADOxx.repodir ADD PRIMARY KEY (id);

ALTER TABLE ADOxx.repodir ADD CONSTRAINT existParent FOREIGN KEY(parent) REFERENCES ADOxx.repodir(id);

ALTER TABLE ADOxx.repoinstance ADD PRIMARY KEY(id);

ALTER TABLE ADOxx.repoinstance ADD CONSTRAINT existDir FOREIGN KEY(dir) REFERENCES ADOxx.repodir(id);

ALTER TABLE ADOxx.instattr_reporef ADD PRIMARY KEY(instance,attribute);

ALTER TABLE ADOxx.message ADD PRIMARY KEY (id,recvid);

ALTER TABLE ADOxx.message ADD CONSTRAINT exRecv FOREIGN KEY(recvid) REFERENCES ADOxx.adouser(userid);

ALTER TABLE ADOxx.filestore ADD PRIMARY KEY (name);

ALTER TABLE ADOxx.rdmmattribute ADD PRIMARY KEY(id);

ALTER TABLE ADOxx.rdmmfacet ADD PRIMARY KEY (attribute,id,libid);


REM *************** Create Indexes ************************

CREATE UNIQUE INDEX ADOxx.class_midid ON ADOxx.class(modclassid,id);

CREATE INDEX ADOxx.class_name ON ADOxx.class(name);

CREATE INDEX ADOxx.class_super ON ADOxx.class(superclass);

CREATE INDEX ADOxx.attr_owia ON ADOxx.attribute (owner,inst_attr);

CREATE INDEX ADOxx.inst_idcl ON ADOxx.instance (id,class);

CREATE INDEX ADOxx.inst_clid ON ADOxx.instance (class,id);

CREATE INDEX ADOxx.inst_moid ON ADOxx.instance (modelid,id);

CREATE INDEX ADOxx.inst_mimo ON ADOxx.instance (modinstid,modelid);

CREATE INDEX ADOxx.inst_name ON ADOxx.instance (name);

CREATE INDEX ADOxx.iadoub_attr ON ADOxx.instattr_double(attribute);

CREATE INDEX ADOxx.iaintint_attr ON ADOxx.instattr_integer(attribute);

CREATE INDEX ADOxx.iastr_attr ON ADOxx.instattr_string(attribute);

CREATE INDEX ADOxx.iastr_inst ON ADOxx.instattr_string(instance);

CREATE INDEX ADOxx.ialong_attr ON ADOxx.instattr_long(attribute);

CREATE INDEX ADOxx.iarec_insattpre ON ADOxx.instattr_record(instance,attribute,preid);

CREATE INDEX ADOxx.iarec_attr ON ADOxx.instattr_record(attribute);

CREATE INDEX ADOxx.iarec_mrecid ON ADOxx.instattr_record(modrecid);

CREATE INDEX ADOxx.iarec_instrecid ON ADOxx.instattr_record(instance,recid);

CREATE INDEX ADOxx.relt_from ON ADOxx.reltype(afrom);

CREATE INDEX ADOxx.relt_to ON ADOxx.reltype(ato);

CREATE INDEX ADOxx.rel_to ON ADOxx.relinstance (ato);

CREATE INDEX ADOxx.rel_from ON ADOxx.relinstance (afrom);

CREATE UNIQUE INDEX ADOxx.usr_name ON ADOxx.adouser(name);

CREATE INDEX ADOxx.lock_id ON ADOxx.adolock (objectid);

CREATE UNIQUE INDEX ADOxx.mod_midlib ON ADOxx.modelthread (modelid,libid);

CREATE INDEX ADOxx.mod_name ON ADOxx.modelthread(name);

CREATE UNIQUE INDEX ADOxx.ver_bidver ON ADOxx.modelversion (threadid, version);

CREATE UNIQUE INDEX ADOxx.name_supergroup ON ADOxx.modelgroup(supergroup,name,libid);

CREATE INDEX ADOxx.id_supermodelgroup ON ADOxx.modelgroup (supergroup);

CREATE UNIQUE INDEX ADOxx.id_groups ON ADOxx.modeltogroup (groupid,modelid);

CREATE INDEX ADOxx.id_modelrefs ON ADOxx.modeltogroup (modelid,groupid);

CREATE INDEX ADOxx.motamo_amidmid ON ADOxx.modeltoamodel (amodelid,modelid);

CREATE UNIQUE INDEX ADOxx.ugrp_name ON ADOxx.usergroup(name);

CREATE UNIQUE INDEX ADOxx.usrtgrp_uidgid ON ADOxx.userstogroups (userid,groupid);

CREATE UNIQUE INDEX ADOxx.perms_ugmgpt ON ADOxx.modelgroupperms (ugroupid,mgroupid,permtype);

CREATE INDEX ADOxx.imr_insatt ON ADOxx.interref (instance,attribute);

CREATE INDEX ADOxx.imr_mona ON ADOxx.interref (tomname);

CREATE INDEX ADOxx.imr_ina ON ADOxx.interref (toiname);

CREATE UNIQUE INDEX ADOxx.rdir_panali ON ADOxx.repodir (parent,name,libid);

CREATE UNIQUE INDEX ADOxx.rpins_name ON ADOxx.repoinstance(name,version,libid);

CREATE INDEX ADOxx.rpins_thve ON ADOxx.repoinstance(thread,version);

CREATE INDEX ADOxx.rpins_mid ON ADOxx.repoinstance(modid);

CREATE INDEX ADOxx.iaref_pname ON ADOxx.instattr_reporef(topname);

CREATE UNIQUE INDEX ADOxx.msg_rvidtime ON ADOxx.message (recvid,stime);

CREATE INDEX ADOxx.rdmma_lid ON ADOxx.rdmmattribute(libid,id);

CREATE INDEX ADOxx.rdmmf_lid ON ADOxx.rdmmfacet(libid);


REM ************* Create views *************************

CREATE VIEW ADOxx.vmodel AS
  SELECT modelversion.id modelid,threadid,name,type,classid,libid
  FROM ADOxx.modelthread,ADOxx.modelversion
  WHERE modelthread.modelid=modelversion.threadid;

CREATE VIEW ADOxx.vinstance AS
  SELECT id,name,class,modelid model,modinstid modid
  FROM ADOxx.instance 
  UNION SELECT id,name,class,0 model,modid
  FROM ADOxx.repoinstance;

CREATE VIEW ADOxx.vimrall AS 
  SELECT modelid,class classid,iaref.attribute attribid,modinstid instid,name iname,tomtype,tomname,tocname,toiname,libid FROM ADOxx.instance,ADOxx.interref iaref WHERE modinstid=iaref.instance OR EXISTS (SELECT iarec.instance
  FROM ADOxx.instattr_record iarec 
  WHERE modrecid=iaref.instance AND iarec.instance=instance.id);

CREATE VIEW ADOxx.vimrsource AS 
  SELECT modelthread.libid,class.modclassid classid,attribid,modelthread.type mtype,modelversion.id modelid,modelthread.name mname,instid,iname,tomtype,tomname,tocname,toiname
  FROM ADOxx.vimrall,ADOxx.modelthread,ADOxx.class,ADOxx.modelversion
  WHERE modelversion.id=vimrall.modelid AND class.id=vimrall.classid AND modelversion.threadid=modelthread.modelid;

CREATE VIEW ADOxx.vrelinmod AS
  SELECT modelid,relinstance.id,reltype,afrom,ato
  FROM ADOxx.relinstance,ADOxx.instance
  WHERE afrom=instance.id;


REM ************* Create triggers ************************

CREATE TRIGGER ADOxx.delModel AFTER DELETE ON ADOxx.modelthread
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.permissions WHERE objectid=:old.modelid;
    DELETE FROM ADOxx.instance WHERE instance.modelid=:old.modelid;
    DELETE FROM ADOxx.modeltoamodel WHERE modeltoamodel.modelid=:old.modelid;
  END;
/

CREATE TRIGGER ADOxx.delVersion AFTER DELETE ON ADOxx.modelversion
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instance WHERE instance.modelid = :old.id;
    DELETE FROM ADOxx.modeltoamodel WHERE modeltoamodel.modelid = :old.id;
  END;
/

CREATE TRIGGER ADOxx.delUser AFTER DELETE ON ADOxx.adouser
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.permissions WHERE permissions.userid = :old.userid;
    DELETE FROM ADOxx.userpref WHERE userpref.userid=:old.userid;
  END;
/

CREATE TRIGGER ADOxx.delAttribute AFTER DELETE ON ADOxx.attribute
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.facet_string WHERE attribute= :old.id;
    DELETE FROM ADOxx.facet_integer WHERE attribute= :old.id;
    DELETE FROM ADOxx.facet_double WHERE attribute= :old.id;
    DELETE FROM ADOxx.instattr_string WHERE attribute= :old.id;
    DELETE FROM ADOxx.instattr_long WHERE attribute= :old.id;
    DELETE FROM ADOxx.instattr_integer WHERE attribute= :old.id;
    DELETE FROM ADOxx.instattr_double WHERE attribute= :old.id;
    DELETE FROM ADOxx.instattr_record WHERE attribute= :old.id;
    DELETE FROM ADOxx.interref WHERE attribute= :old.id;
  END;
/

CREATE TRIGGER ADOxx.delClass AFTER DELETE ON ADOxx.class
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instattr_string WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_long WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_integer WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_double WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_record WHERE instance=:old.id;
    DELETE FROM ADOxx.interref WHERE instance=:old.id;
    DELETE FROM ADOxx.attribute WHERE owner= :old.id;
    DELETE FROM ADOxx.reltype WHERE afrom= :old.id OR ato= :old.id;
    DELETE FROM ADOxx.instance WHERE class= :old.id;
  END;
/

CREATE TRIGGER ADOxx.delRelType AFTER DELETE ON ADOxx.reltype
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instattr_string WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_long WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_integer WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_double WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_record WHERE instance=:old.id;
    DELETE FROM ADOxx.interref WHERE instance=:old.id;
    DELETE FROM ADOxx.attribute WHERE owner= :old.id;
    DELETE FROM ADOxx.relinstance WHERE reltype=:old.id;
  END;
/

CREATE TRIGGER ADOxx.delInstance AFTER DELETE ON ADOxx.instance
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instattr_string WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_long WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_integer WHERE instance = :old.id;
    DELETE FROM ADOxx.instattr_double WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_record WHERE instance = :old.id;
    DELETE FROM ADOxx.instattr_reporef WHERE instance = :old.id;
    DELETE FROM ADOxx.relinstance WHERE afrom = :old.id;
    DELETE FROM ADOxx.relinstance WHERE ato = :old.id;
    DELETE FROM ADOxx.interref WHERE instance = :old.modinstid;
  END;
/

CREATE TRIGGER ADOxx.delRInstance AFTER DELETE ON ADOxx.repoinstance
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instattr_string WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_long WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_integer WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_double WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_record WHERE instance=:old.id;
    DELETE FROM ADOxx.interref WHERE instance = :old.modid;
  END;
/

CREATE TRIGGER ADOxx.delRelation AFTER DELETE ON ADOxx.relinstance
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instattr_string WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_long WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_integer WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_double WHERE instance=:old.id;
    DELETE FROM ADOxx.instattr_record WHERE instance=:old.id;
  END;
/

CREATE TRIGGER ADOxx.delRecord BEFORE DELETE ON ADOxx.instattr_record
  FOR EACH ROW BEGIN
    DELETE FROM ADOxx.instattr_string WHERE instance=:old.recid;
    DELETE FROM ADOxx.instattr_long WHERE instance= :old.recid;
    DELETE FROM ADOxx.instattr_integer WHERE instance=:old.recid;
    DELETE FROM ADOxx.instattr_double WHERE instance=:old.recid;
    DELETE FROM ADOxx.instattr_reporef WHERE instance=:old.recid;
    DELETE FROM ADOxx.interref WHERE instance=:old.modrecid;
  END;
/
