## LOAD_LIBRARY

Loads an application library with the specified ID, allowing to execute some of
the other library manipulation commands.



### Syntax

```AdoScript
CC "Core" LOAD_LIBRARY libid:idValue [ write-protected:boolValue ]
# --> RESULT ecode:intValue applibid:idValue bplibid:idValue welibid:idValue
```



### Parameters

* `libid` (idValue) - The ID of the application library to load. This has to
  be the ID of the application library, not of a dynamic or static library!
* `write-protected` (boolValue, optional) - Specifies that the library is not
  going to be changed, but just loaded for reading purposes. However, a library
  loaded with active "write protected" flag can still be modified. The only
  difference is that it is not possible that a library is loaded by two users
  without "write protected" flag at a time. The default is true.



### Returns

* `ecode` (intValue) - Contains the error code or 0 in case of success.
* `applibid` (idValue) - the ID of the loaded application library.
* `bplibid` (idValue) - the ID of the corresponding dynamic (bp) library.
* `welibid` (idValue) - the ID of the corresponding static (we) library.



### Remarks

Only one library can be loaded at a time. If there is a library currently
loaded, then this command will fail (non-0 ecode). To load a different library
the current one must first be discarded using the `"Core" DISCARD_LIBRARY`
command.

In the Modelling Toolkit the library of the logged in user is loaded by
default. Since the library can not be discarded (unloaded) in the Modelling
Toolkit the `"Core" LOAD_LIBRARY` command is useless there.



### Examples

Load a library, get the ID of a class and then discard the library again.

```AdoScript
# Get the ID of the library to be accessed.
CC "Core" GET_LIBRARY_ID libname:("ADOxx 1.8 Experimentation Library")
SETL idCurAppLib:(libid)
# Load the library, so that it can be accessed.
CC "Core" LOAD_LIBRARY libid:(idCurAppLib)
SETL idCurBpLib:(bplibid)
# Get the desired class ID
CC "Core" GET_CLASS_ID classname:("__D-construct__")
SETL idClass:(classid)
# Discard the library since we're done modifying it.
CC "Core" DISCARD_LIBRARY
# Show the ID of the class.
CC "AdoScript" INFOBOX ("ID of class: " + STR idClass)
```

Create a new class in the dynamic library through the Development Toolkit.

```AdoScript
# Get the ID of the library to be modified.
CC "Core" GET_LIBRARY_ID libname:("ADOxx 1.8 Experimentation Library")
SETL idCurAppLib:(libid)
# Load the library, so that it can be modified.
CC "Core" LOAD_LIBRARY libid:(idCurAppLib)
SETL idCurBpLib:(bplibid)
# Get the ID of the super class for the new class.
CC "Core" GET_CLASS_ID classname:("__D-construct__")
SETL idSuperClass:(classid)
# Create the new class.
CC "Core" CREATE_CLASS libid:(idCurBpLib)
    superid:(idSuperClass)
    classname:"MyTestClass"
SETL idClass:(classid)
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
# Discard the library since we're done modifying it.
CC "Core" DISCARD_LIBRARY
# Show the ID of the newly created class.
CC "AdoScript" INFOBOX ("ID of created class: " + STR idClass)
```

Shows all available application libraries and information about the currently
loaded application library. Afterwards it discards any currently loaded library
and loads another library.

```AdoScript
# Show all available application libraries
SETL sLibs:("")
SETL sAvailLibs:("")
CC "Core" GET_ALL_APP_LIBRARIES
FOR id in:(applibids) {
  CC "Core" GET_LIB_NAME libid:(VAL id)
  CC "Core" GET_LIB_ID libname:(libname)
  SET sLibs:(sLibs + STR libid + ":\t" + libname + "\n")
  SET sAvailLibs:(sAvailLibs + libname + ";")
}
CC "AdoScript" VIEWBOX text:(sLibs) title:"GET_ALL_LIBS"

# Show the current application library
CC "Core" GET_CURRENT_LIBRARIES
SETL idApplib:(applibid)
CC "AdoScript" INFOBOX ("Current application library:\n" + applib)
    title:"GET_CURRENT_LIBS"

# Discard the current library if one is loaded
IF (idApplib > 0) {
  CC "Core" DISCARD_LIB
  CC "AdoScript" INFOBOX ("Discarded Lib ID: " + STR libid) title:"DISCARD_LIB"
}

# Load another library
CC "AdoScript" LISTBOX entries:(sAvailLibs) toksep:";"
    title:"Load Application Library" oktext:"Select"
    boxtext:"Choose Library ID:"
IF (endbutton = "ok") {
  SETL idApplib:(-1)
  CC "Core" GET_LIB_ID libname:(selection)
  CC "Core" LOAD_LIB libid:(libid)
  SETL idApplib:(applibid)
  CC "AdoScript" INFOBOX (errtext + "\nAppLib-ID: " + (STR idApplib))
      title:"LOAD_LIB"
}
```



### See Also

* `"Core" GET_ALL_APP_LIBRARIES`
* `"Core" GET_CURRENT_LIBRARIES`
* `"Core" GET_LIBRARY_ID`
* `"Core" SAVE_LIBRARY`
* `"Core" DISCARD_LIBRARY`



### Versions and Changes

Available since ADOxx 1.3
* Until ADOxx 1.8.0 it was named `"Core" LOAD_LIB`.

*ADOxx 1.8.0*
* This command name has been updated to be consistent with the other ones. The
  old command name `"Core" LOAD_LIB` is still available for backwards
  compatibility. However, it is recommended to use the new name instead.
