## CREATE_OBJ

Creates a new object / instance of a certain class in a certain model.



### Syntax

```AdoScript
CC "Core" CREATE_OBJ modelid:idValue classid:idValue [ objname:strValue ]
# --> RESULT ecode:intValue objid:idValue
```



### Parameters

* `modelid` (idValue) - The ID of the model in which the object should be
  created.
* `classid` (idValue) - The ID of the class of which an object should be
  created.
* `objname` (strValue, optional) - The name of the new object. The default is
  based on the class name and the ID of the object (`[classname]-[objid]`).



### Returns

* `ecode` (intValue) - Contains the error code or 0 in case of success.
* `objid` (idValue) - The ID of the created object.



### Remarks

The model where the object is to be created must be loaded. Also when `objname`
is specified, then no other object of the same class with that name may exist
in the model model.

Created objects are only persisted into the database once the model is saved.



### Examples

Create a new object of the class "MyTestClass" in the currently active model.

```AdoScript
# Get the currently active model.
CC "Modeling" GET_ACT_MODEL
SETL idModel:(modelid)
# Get the class ID of "MyTestClass".
CC "Core" GET_CLASS_ID classname:("MyTestClass")
SETL idClass:(classid)
# Create a new object of the class.
CC "Core" CREATE_OBJ modelid:(idModel)
    classid:(idClass)
SETL idNewObject:(objid)
```

Create 10 new objects of the class "MyTestClass" in the currently active model
with their name being the class name with their number appended. It also
updates the status bar with progress information and positions the objects in a
square-like shape. Note: positioning more than 10 objects causes a massive
slowdown!

```AdoScript
# Configuration variables.
SETL iAmount:(10)
SETL sClassName:("MyTestClass")

# Helper variables and loaded values.
SETL aNewObjs:(array(iAmount))
SETL iColumnCount:(INT (sqrt(iAmount)))
CC "Modeling" GET_ACT_MODEL
SETL idModel:(modelid)
IF (idModel < 0) {
  CC "AdoScript" ERRORBOX ("No active model!")
  EXIT
}
CC "Core" GET_CLASS_ID classname:(sClassName)
IF (ecode) {
  CC "AdoScript" ERRORBOX ("Could not find class " + sClassName)
  EXIT
}
SETL idClass:(classid)

# Create the objects in the model
FOR iIndex from:(0) to:(iAmount - 1) {
  CC "Core" CREATE_OBJ modelid:(idModel)
      classid:(idClass)
      objname:(sClassName + STR (iIndex + 1))
  SET aNewObjs[iIndex]:(objid)
  CC "Application" SET_STATUS ("Object created: " + STR (iIndex + 1) + "/" + STR iAmount)
}

# Position the newly created objects in a square-like shape.
CC "Modeling" REBUILD_DRAWING_AREA
FOR iIndex from:(0) to:(iAmount - 1) {
  SET mX:(2cm + CM ((iIndex - (floor(iIndex / iColumnCount)) * iColumnCount) * 4))
  SET mY:(2cm + CM (floor(iIndex / iColumnCount) * 4))
  CC "Modeling" SET_OBJ_POS objid:(aNewObjs[iIndex]) x:(mX) y:(mY)
  CC "Application" SET_STATUS ("Objekt positioned: " + STR (iIndex + 1) + "/" + STR iAmount)
}
```



### See Also

* `"Core" LOAD_MODEL`
* `"Modeling" OPEN`
* `"Core" SAVE_MODEL`
* `"Core" DELETE_OBJS`



### Versions and Changes

Available since ADOxx 1.3
