#!/bin/bash

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Asks the user whether to uninstall the Wine. If so, then the Wine is
# uninstalled.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.

# Notes about some specifics of the script/s:
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# The following establishes the absolute path to this scripts directory.
ABSOLUTE_SCRIPT_DIRECTORY=$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")"&>/dev/null && pwd)
# Process passed arguments and load various relevant variables and functions.
source "${ABSOLUTE_SCRIPT_DIRECTORY}/common_script.sh"



# Check if Wine is installed.
command -v wine64 &>/dev/null
wine64_command_available=$?
if [ ${wine64_command_available} -eq 0 ]
then
    echo "Wine found."
else
    echo "Wine is not installed."
fi



# If Wine is installed, then ask the user whether it should be uninstalled.
uninstall_wine=1
if [ ${wine64_command_available} -eq 0 ] && [ ${force_operation} -ne 0 ]
then
    echo ${warning_style}"Uninstalling Wine will prevent the use of other ADOxx based tools and other"${default_style}
    echo ${warning_style}"(Windows) applications that rely on Wine. If you have other ADOxx based"${default_style}
    echo ${warning_style}"tools that you want to use, then this is not recommended."${default_style}
    prompt_user_default_no ${input_style}"Should Wine be uninstalled? [y/N] or press CTRL-C to abort."${default_style}
    uninstall_wine=$?
fi



# Uninstall Wine if so desired.
if [ ${uninstall_wine} -eq 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Uninstalling Wine."
    # This is gathered before uninstalling wine to remove a possible symbolic link that was created by us later.
    wine64_location="$(which wine64)"
    # This used to work, but then after some wine updates the wine and winehq-stable branches caused a crash when trying to start DevTk / ModTk.
    #sudo ${linux_package_manager} remove wine
    sudo ${linux_package_manager} remove winehq-staging
    # Also remove the symbolic link for wine64 if it exists.
    # -h tests for a symbolic link, ! -e tests if (target) file doesn't exist.
    if [ -h $wine64_location ] && [ ! -e $wine64_location ]
    then
        echo "Removing link ${wine64_location} (probably created during the installation)."
        sudo rm "$wine64_location"
    fi
else
    echo "Wine is NOT being uninstalled."
fi



# Check if Wine has been uninstalled and "return" 0 (true) or 1 (false).
command -v wine64 &>/dev/null
wine64_command_available=$?
if [ ${uninstall_wine} -eq 0 ] && [ ${wine64_command_available} -eq 0 ]
then
    echo ${error_style}"FAILED uninstalling Wine."
    echo "Please uninstall Wine manually."${default_style}
    exit 1
else
    exit 0
fi
