#!/bin/bash

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Asks the user whether to delete the database file. If so, then the database
# file is deleted. This includes all the models that have been stored there.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.
# wine_prefix_dir - The Wine prefix directory.
# database_name - The name of the database, which is also the name of the file.

# Notes about some specifics of the script/s:
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# The following establishes the absolute path to this scripts directory.
ABSOLUTE_SCRIPT_DIRECTORY=$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")"&>/dev/null && pwd)
# Process passed arguments and load various relevant variables and functions.
source "${ABSOLUTE_SCRIPT_DIRECTORY}/common_script.sh"



# Check if the database already exist.
# The logic has to follow the same logic that is used by ADOxx.
# The first test is SIMPLE, assuming that the file is in a very specific
# location. This SHOULD be the most likely case.
user_folder="$(id -un)"
database_file="${wine_prefix_dir}/drive_c/users/${user_folder}/AppData/Roaming/ADOxx/sqlitedbs/${database_name}.sqlite3"
[ -f "${database_file}" ]
database_exists=$?
if [ ${database_exists} -eq 0 ]
then
    echo "Database file found: ${database_file}"
fi

# If the previous check did not find the database file, then we "ask" Wine
# instead. There are cases where the file can be located in a different place:
# * Some Wine versions use "Application Data/" instead "AppData/Roaming/".
# * Some Wine versions do not use the system user's name for the user name
#   in the Wine prefix.
if [ ${database_exists} -ne 0 ]
then
    # First check if the Wine prefix folder exists and Wine is installed.
    if [ ! -d "${wine_prefix_dir}" ]
    then
        echo ${warning_style}"Wine prefix not available. So the database is most likely gone too."${default_style}
        exit 0
    fi
    command -v wine64 &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ${error_style}"Wine not available. Unable to determine exact place of the database."
        echo "Please install Wine first and ensure that the wine64 command is available."${default_style}
        exit 1
    fi
    # We get the value of the %APPDATA% environment variable.
    # Note: The value assigned here ends with a CR (\r) character, because Windows!
    wine_prefix_addpata_dir="$(LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine64 cmd /c "ECHO %APPDATA%" 2>/dev/null)"
    # And then we use winepath.exe to transform it to a unix path.
    # Note: ${wine_prefix_addpata_dir::-1} removes the last character (the CR) from the variable.
    wine_prefix_appdata_dir="$(LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine64 winepath.exe "${wine_prefix_addpata_dir::-1}" 2>/dev/null)"
    database_file="${wine_prefix_appdata_dir}/ADOxx/sqlitedbs/${database_name}.sqlite3"
    # Check again fi the database file exists.
    [ -f "${database_file}" ]
    database_exists=$?
    if [ ${database_exists} -eq 0 ]
    then
        echo "Database file found: ${database_file}"
    fi

    # Approaches previously tried:

    # Using compgen to use all the user in the Wine prefix.
    # This also used an amount_of_database_files_found variable and a lot of
    # other code later on.
    #wineprefix_users_folder="${wine_prefix_dir}/drive_c/users"
    #database_file_in_user_folder="AppData/Roaming/ADOxx/sqlitedbs/${database_name}.sqlite3"
    #found_database_files="$(compgen -G "${wineprefix_users_folder}/*/${database_file_in_user_folder}" 2>/dev/null)"

    # Using find to search for the database file in the Wine prefix folder.
    # This also used an amount_of_database_files_found variable and a lot of
    # other code later on.
    #database_file_name="${database_name}.sqlite3"
    #wineprefix_users_folder="${wine_prefix_dir}/drive_c/users"
    #found_database_files="$(find "${wineprefix_users_folder}" -name "${database_file_name}" 2>/dev/null)"
fi

# If we still didn't find the database file then we assume it is missing.
if [ ${database_exists} -ne 0 ]
then
    echo "Database missing."
fi



# If a database file exists, then ask the user whether it should be deleted.
delete_database=1
if [ ${database_exists} -eq 0 ] && [ ${force_operation} -ne 0 ]
then
    echo ${warning_style}"Deleting the database file will also delete all models of the tool."${default_style}
    echo ${warning_style}"Make sure you no longer need them or that they have been exported as"${default_style}
    echo ${warning_style}".adl / .xml files first."${default_style}
    echo "Database file to delete: ${database_file}"
    prompt_user_default_no ${input_style}"Should the database be deleted? [y/N] or press CTRL-C to abort."${default_style}
    delete_database=$?
fi



# Delete the database file if so desired.
if [ ${delete_database} -eq 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Deleting database."
    rm -f "${database_file}"
else
    echo "Database file is NOT being deleted."
fi



# Check if the database file has been removed and "return" 0 (true) or 1 (false).
if [ ${delete_database} -eq 0 ] && [ -f "${database_file}" ]
then
    echo ${error_style}"FAILED deleting database file."
    echo "Please delete the database file manually."${default_style}
    exit 1
else
    exit 0
fi
