#!/bin/bash

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Copies the shortcuts for the tool.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# tool_name - The name of the tool to be installed.
# tool_version - The version of the tool to be installed.
# wine_prefix_dir - The Wine prefix directory.
# tool_installation_dir - The directory to install the tool to inside the Wine
#   prefix.

# Notes about some specifics of the script/s:
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# The following establishes the absolute path to this scripts directory.
ABSOLUTE_SCRIPT_DIRECTORY="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")"&>/dev/null && pwd)"
# Process passed arguments and load various relevant variables and functions.
source "${ABSOLUTE_SCRIPT_DIRECTORY}/common_script.sh"

# Helper variables.
full_tool_installation_dir="${wine_prefix_dir}/drive_c/${tool_installation_dir}"
linux_icons_directory="${HOME}/.local/share/icons/"
linux_application_directory="${HOME}/.local/share/applications/"
linux_desktop_directory="${HOME}/Desktop/"

modtk_shortcut_icon="${tool_name} ${tool_version}.png"
devtk_shortcut_icon="${tool_name} ${tool_version}_devtk.png"



# Check which of the shortcuts to copy are available.
# These have to follow the same logic as they are generated by the PDP.
if [ -n "${tool_modtk_shortcut_suffix}" ]
then
    modtk_shortcut_name="${tool_name} ${tool_version} ${tool_modtk_shortcut_suffix}.desktop"
else
    modtk_shortcut_name="${tool_name} ${tool_version}.desktop"
fi
[ -f "${ABSOLUTE_SCRIPT_DIRECTORY}/../${modtk_shortcut_name}" ]
modtk_shortcut_provided=$?

devtk_shortcut_name="${tool_name} ${tool_version} Development Toolkit.desktop"
[ -f "${ABSOLUTE_SCRIPT_DIRECTORY}/../${devtk_shortcut_name}" ]
devtk_shortcut_provided=$?

installation_dir_shortcut_name="${tool_name} ${tool_version} Installation Folder"



# Check if any of the shortcuts already exist.
if [ ${modtk_shortcut_provided} -eq 0 ] && [ -f "${linux_application_directory}/${modtk_shortcut_name}" ]
then
    echo "${modtk_shortcut_name} already exists in application directory."
fi
if [ ${modtk_shortcut_provided} -eq 0 ] && [ -f "${linux_desktop_directory}/${modtk_shortcut_name}" ]
then
    echo "${modtk_shortcut_name} already exists in desktop directory."
fi

if [ ${devtk_shortcut_provided} -eq 0 ] && [ -f "${linux_application_directory}/${devtk_shortcut_name}" ]
then
    echo "${devtk_shortcut_name} already exists in application directory."
fi
if [ ${devtk_shortcut_provided} -eq 0 ] && [ -f "${linux_desktop_directory}/${devtk_shortcut_name}" ]
then
    echo "${devtk_shortcut_name} already exists in desktop directory."
fi

if [ -L "${linux_application_directory}/${installation_dir_shortcut_name}" ]
then
    echo "Link to ${installation_dir_shortcut_name} already exists in application directory."
fi



# Copy the available shortcuts.
# Whether to overwrite existing files or not is handled by the copy command.
if [ ${modtk_shortcut_provided} -eq 0 ]
then
    echo "Copying ${modtk_shortcut_name} shortcuts"
    echo "-- shortcut icon ${linux_icons_directory}/${modtk_shortcut_icon}"
    cp "${ABSOLUTE_SCRIPT_DIRECTORY}/../${modtk_shortcut_icon}" "${linux_icons_directory}/${modtk_shortcut_icon}"
    if [ -d "${linux_application_directory}" ]
    then
        echo "-- to ${linux_application_directory}"
        cp "${ABSOLUTE_SCRIPT_DIRECTORY}/../${modtk_shortcut_name}" "${linux_application_directory}/"
        sudo chmod +x "${linux_application_directory}/${modtk_shortcut_name}"
    fi
    if [ -d "${linux_desktop_directory}" ]
    then
        echo "-- to ${linux_desktop_directory}"
        cp "${ABSOLUTE_SCRIPT_DIRECTORY}/../${modtk_shortcut_name}" "${linux_desktop_directory}/"
        sudo chmod +x "${linux_desktop_directory}/${modtk_shortcut_name}"
    fi
fi

if [ ${devtk_shortcut_provided} -eq 0 ]
then
    echo "Copying ${devtk_shortcut_name} shortcuts"
    echo "-- shortcut icon ${linux_icons_directory}/${devtk_shortcut_icon}"
    cp "${ABSOLUTE_SCRIPT_DIRECTORY}/../${devtk_shortcut_icon}" "${linux_icons_directory}/${devtk_shortcut_icon}"
    if [ -d "${linux_application_directory}" ]
    then
        echo "-- to ${linux_application_directory}"
        cp "${ABSOLUTE_SCRIPT_DIRECTORY}/../${devtk_shortcut_name}" "${linux_application_directory}/"
        sudo chmod +x "${linux_application_directory}/${devtk_shortcut_name}"
    fi
    if [ -d "${linux_desktop_directory}" ]
    then
        echo "-- to ${linux_desktop_directory}"
        cp "${ABSOLUTE_SCRIPT_DIRECTORY}/../${devtk_shortcut_name}" "${linux_desktop_directory}/"
        sudo chmod +x "${linux_desktop_directory}/${devtk_shortcut_name}"
    fi
fi

# Updating the installation directory link.
if [ -L "${linux_application_directory}/${installation_dir_shortcut_name}" ]
then
    echo "Removing existing link to installation directory: ${installation_dir_shortcut_name}"
    rm -r "${linux_application_directory}/${installation_dir_shortcut_name}"
fi
echo "Creating link to installation directory: ${installation_dir_shortcut_name}"
ln -s "${full_tool_installation_dir}/" "${linux_application_directory}/${installation_dir_shortcut_name}"



# Check if the shortcut files exist and "return" 0 (true) or 1 (false).
shortcuts_missing=1

if [ ${modtk_shortcut_provided} -eq 0 ] && [ ! -f "${linux_application_directory}/${modtk_shortcut_name}" ]
then
    echo "${modtk_shortcut_name} wasn't copied to application directory."
    shortcuts_missing=0
fi
if [ ${modtk_shortcut_provided} -eq 0 ] && [ ! -f "${linux_desktop_directory}/${modtk_shortcut_name}" ]
then
    echo "${modtk_shortcut_name} wasn't copied to desktop directory."
    shortcuts_missing=0
fi

if [ ${devtk_shortcut_provided} -eq 0 ] && [ ! -f "${linux_application_directory}/${devtk_shortcut_name}" ]
then
    echo "${devtk_shortcut_name} wasn't copied to application directory."
    shortcuts_missing=0
fi
if [ ${devtk_shortcut_provided} -eq 0 ] && [ ! -f "${linux_desktop_directory}/${devtk_shortcut_name}" ]
then
    echo "${devtk_shortcut_name} wasn't copied to desktop directory."
    shortcuts_missing=0
fi

if [ ! -h "${linux_application_directory}/${installation_dir_shortcut_name}" ]
then
    echo "Link to ${installation_dir_shortcut_name} wasn't created."
    shortcuts_missing=0
fi

if [ ${shortcuts_missing} -eq 0 ]
then
    echo ${error_style}"FAILED copying (some) shortcuts."
    echo "Please copy and configure missing shortcuts manually."${default_style}
    exit 1
else
    exit 0
fi
