#!/bin/bash

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Sets the relevant license files in the tool installation directory. If not
# enough information is available, then the user is prompted to provide their
# license information.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# force_operation - Performs the operation independent of checks / user inputs.
# wine_prefix_dir - The Wine prefix directory.
# tool_installation_dir - The directory to install the tool to inside the Wine
#   prefix.
# license_key - The license key to use. If it is empty, then the user is
#   prompted to provide their own license details.

# Notes about some specifics of the script/s:
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# The following establishes the absolute path to this scripts directory.
ABSOLUTE_SCRIPT_DIRECTORY="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")"&>/dev/null && pwd)"
# Process passed arguments and load various relevant variables and functions.
source "${ABSOLUTE_SCRIPT_DIRECTORY}/common_script.sh"

# Helper variables.
full_tool_installation_dir="${wine_prefix_dir}/drive_c/${tool_installation_dir}"



# Check if the license files already exist.
[ -f "${full_tool_installation_dir}/alicdat.ini" ] && [ -f "${full_tool_installation_dir}/license.ini" ]
license_data_exists=$?
if [ ${license_data_exists} -eq 0 ]
then
    echo "License data already exists."
else
    echo "License data missing."
fi



# If the license data already exists, then prompt the user whether they should
# be replaced.
if [ ${license_data_exists} -eq 0 ]
then
    prompt_user_default_no ${input_style}"Should the license information be replaced? [y/N] or press CTRL-C to abort."${default_style}
    if [ $? -eq 0 ]
    then
        echo "Removing license data."
        rm -f "${full_tool_installation_dir}/alicdat.ini"
        rm -f "${full_tool_installation_dir}/license.ini"
        license_data_exists=1
    fi
fi



# Create the license files. Depending on whether license_key is provided this
# is entirely automated or prompts the user to specify their license data.
if [ ${license_data_exists} -ne 0 ] || [ ${force_operation} -eq 0 ]
then
    echo "Setting license data."
    if [ "${license_key}" = "" ]
    then
        echo ${warning_style}"Please wait for the user prompt to show up and enter the license information"${default_style}
        echo ${warning_style}"that you received via ADOxx.org. You can paste a copied text using CTRL+v"${default_style}
        echo ${warning_style}"If the window seems hidden, then look for the wine64-preloader window."${default_style}
        LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine64 "C:/${tool_installation_dir}/asetlic.exe" "-p${full_tool_installation_dir}/license.ini" "-L${full_tool_installation_dir}/alicdat.ini"
    else
        LANG=en_US WINEARCH=win64 WINEPREFIX=${wine_prefix_dir} WINEDEBUG=-all wine64 "C:/${tool_installation_dir}/asetlic.exe" "-p${full_tool_installation_dir}/license.ini" "-L${full_tool_installation_dir}/alicdat.ini" -cBOCGmbH "-l${license_key}"
    fi
fi



# Check if the license files exist and "return" 0 (true) or 1 (false).
if [ ! -f "${full_tool_installation_dir}/alicdat.ini" ] || [ ! -f "${full_tool_installation_dir}/license.ini" ]
then
    echo ${error_style}"FAILED configuring license."
    echo "Please configure license manually."${default_style}
    exit 1
else
    exit 0
fi
