#!/bin/bash

# Version: 1.0.0
# Date: 2023-10-12
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# This file contains the installation procedure for the tool. It uses many
# other script files to get the job done. It is provided as a separate file
# to allow the main installation file seen by the user (install_on_linux.sh)
# to perform updates to these installation scripts, e.g. downloading newer
# versions from the internet.

# The components rely on each other, requiring a specific sequence, i.e.:
# Creating the database -> requires the license data -> requires the tool to
# be installed -> requires a Wine prefix -> requires Wine.
# Therefore, each step of the installation returns whether it has been
# successful (default when no return value) or not and if it was not
# successful, then the installation aborts.

# This script uses set_variables.sh to set various variables that are used.
# The relevant variables are:
# tool_name - The name of the tool to be installed.
# tool_version - The version of the tool to be installed.

# Notes about some specifics of the script/s:
# * ${@} - represents all the parameters / arguments that have been passed to
#   this script.
# * Calling a script via "./filename.sh" executes them in a separate process
#   (at least in zsh), so they should have their own local (environment)
#   variables.
# * Adding a script via the source commands behaves as if the code from the
#   other script was part of this script, similar to #include in C/C++.



# The following establishes the absolute path to this scripts directory.
ABSOLUTE_SCRIPT_DIRECTORY="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")"&>/dev/null && pwd)"
# Process passed arguments and load various relevant variables and functions.
source "${ABSOLUTE_SCRIPT_DIRECTORY}/common_script.sh"

grey=''
if [ ${no_style} -ne 0 ]
then
    grey=$'\e[37m'
fi



# Print general information for the user.
echo
echo "-----------------------------------------------------------------   ${warning_style}*        ${grey}#"${default_style}
echo "|                                                                  ${warning_style}***     ${grey}###"${default_style}
echo "|         ADOxx Installer for Linux                               ${warning_style}*****  ${grey}#####"${default_style}
echo "|         (Experimental Release, v1.0.2)                               ${grey}#######"${default_style}
echo "|                                                                    ${grey}#########"${default_style}
echo "-----------------------------------------------------------------  ${grey}###########"${default_style}
echo
echo "This script installs ADOxx or an ADOxx based tool on your 64-bit Linux system."
echo ${warning_style}"Before starting the installation, make sure that your system is up-to-date."${default_style}
echo "Provide --nostyle argument to the script to omit the use of styles and colors."
echo "Check the README.md and README-ADOxx.md for further details."
echo ${warning_style}"You can stop the execution at any point by pressing CTRL+C."${default_style}
echo "------------------------------------------------------------------------------"
echo
echo "Tool to be installed: ${tool_name} ${tool_version}"
echo
echo "What is required:"
echo "Internet connection - some resources are downloaded from the internet."
echo "Administrative password - some steps can require administrative privileges."
echo "Manual steps - in some cases manual steps are needed."
echo "------------------------------------------------------------------------------"
read -p "${input_style}Hit ENTER to continue"${default_style}
echo

echo "Components installed / set up with this script (if they are missing):"
echo "1. Install Wine."
echo "2. Set up Wine prefix."
echo "3. Set up tool directory. (uses unzip)"
echo "4. Configure license."
echo "5. Create database."
echo "6. Copy shortcuts."
echo "------------------------------------------------------------------------------"
echo ${warning_style}"Pressing CTRL+C to abort the installation."${default_style}
read -p "${input_style}Hit ENTER to install / set up the above components."${default_style}
echo



# Check if user is root. Being root can mess up the installation, so abort.
if [ $(whoami) = "root" ]
then
    echo ${error_style}"Do not execute this script as root (sudo)!"${default_style}
    exit
fi



# Perform the installation steps in their sequence.
echo "=== 1. Install Wine."
"${ABSOLUTE_SCRIPT_DIRECTORY}/i1_install_wine.sh" "${@}"
if [ $? -ne 0 ]
then
    prompt_user_default_no ${input_style}"Continue with the installation? [y/N]"${default_style}
    if [ $? -ne 0 ]
    then
        exit
    fi
fi
echo

echo "=== 2. Set up Wine prefix."
"${ABSOLUTE_SCRIPT_DIRECTORY}/i2_create_wineprefix.sh" "${@}"
if [ $? -ne 0 ]
then
    prompt_user_default_no ${input_style}"Continue with the installation? [y/N]"${default_style}
    if [ $? -ne 0 ]
    then
        exit
    fi
fi
echo

echo "=== 3. Set up tool directory."
"${ABSOLUTE_SCRIPT_DIRECTORY}/i3_copy_tool.sh" "${@}"
if [ $? -ne 0 ]
then
    prompt_user_default_no ${input_style}"Continue with the installation? [y/N]"${default_style}
    if [ $? -ne 0 ]
    then
        exit
    fi
fi
echo

echo "=== 4. Configure license."
"${ABSOLUTE_SCRIPT_DIRECTORY}/i4_configure_license.sh" "${@}"
if [ $? -ne 0 ]
then
    prompt_user_default_no ${input_style}"Continue with the installation? [y/N]"${default_style}
    if [ $? -ne 0 ]
    then
        exit
    fi
fi
echo

echo "=== 5. Create database."
"${ABSOLUTE_SCRIPT_DIRECTORY}/i5_create_database.sh" "${@}"
if [ $? -ne 0 ]
then
    prompt_user_default_no ${input_style}"Continue with the installation? [y/N]"${default_style}
    if [ $? -ne 0 ]
    then
        exit
    fi
fi
echo

echo "=== 6. Copy shortcuts."
"${ABSOLUTE_SCRIPT_DIRECTORY}/i6_copy_shortcuts.sh" "${@}"
if [ $? -ne 0 ]
then
    prompt_user_default_no ${input_style}"Continue with the installation? [y/N]"${default_style}
    if [ $? -ne 0 ]
    then
        exit
    fi
fi
echo

echo "=== Installation script finished ==="
echo
