#!/bin/bash

# Version: 1.0.0
# Date: 2023-10-06
# Authors: Christian Muck, Patrik Burzynski, Wilfrid Utz
# Installation scripts: MIT License, (C) Copyright 2023 OMiLAB

# Processes arguments and sets various variables based on those arguments. It
# also provides several variables, like loading variables that hold the tool's
# details (tool_config.env). To use this script simply call it from a
# parent script with the `source` command.

# It also contains various useful functions that are used in other scripts.

# The basic arguments are:
# --nostyle - Sets nostyle variable to 0 (true). Default is 1 (false).
# --forceop - Sets force_operation variable to 0 (true). Default is 1 (false).

# It also has a special argument, which allows to set any variable to a value:
# --d-VARIABLE=VALUE - Sets the value of the VARIABLE to VALUE. For example
#   --d-database=adoxx18 will set the variable database to the value adoxx18.
#   If VALUE contains spaces they have to be escaped (using \ ) or just put the
#   VALUE in double quotes, like --d-somevar="value with spaces"



# The following establishes the absolute path to this scripts directory.
ABSOLUTE_SCRIPT_DIRECTORY=$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")"&>/dev/null && pwd)



# Simple function that prompts the user a "yes or no" question, with defaulting
# to "no". The "return value" $? is 0 for yes or non-0 for no.
function prompt_user_default_no {
    local tempval=""
    if [ $# -gt 0 ]
    then
        read -p "${*} " tempval
    else
        read -p "Yes or no? (default: No) " tempval
    fi
    tempval=$(echo ${tempval} | tr '[:upper:]' '[:lower:]')
    [ -n ${tempval} ] && [[ ${tempval} = y* ]]
}



# Load the tool's details as variables so they can be used in the scripts.
if [ -f "${ABSOLUTE_SCRIPT_DIRECTORY}/../tool_config.env" ]
then
    source "${ABSOLUTE_SCRIPT_DIRECTORY}/../tool_config.env"
fi

# Variables that can be set via arguments.
no_style=1
force_operation=1

# Variables for styling output.
    error_style=''     # Used for errors.
    input_style=''     # Used for messages that should get attention.
    warning_style=''   # Used to indicate user input.
    default_style=''   # Default style of text.
# colors not used:
#    green_style=$'\e[1;32m'
#    magenta_style=$'\e[1;35m'
#    cyan_style=$'\e[1;36m'

# Variables about the Linux distribution and package manager
supported_package_managers=("apt" "apt-get" "dnf" "yum" "zypp")
# Not supported because the commands for installing / removing are different: "apk" "emerge" "pacman"
#linux_distribution=
#linux_distribution_version=
linux_package_manager=
for a_package_manager in ${supported_package_managers[@]}
do
    command -v ${a_package_manager} &>/dev/null
    if [ $? -eq 0 ]
    then
        linux_package_manager=${a_package_manager}
        break
    fi
done



# Process passed arguments and set variables accordingly.
# Performed after loading tool_config.env to allow overwriting them.
for argvar in ${@}
do
    if [ "${argvar}" = "--nostyle" ]
    then
        no_style=1
    elif [ "${argvar}" = "--forceop" ]
    then
        force_operation=0
    elif [ "${argvar[1,4]}" = "--d-" ] && [[ "${argvar}" =~ "=" ]]
    then
        # This allows to set other variables in the script passed as arguments.
        eval $(echo ${argvar[5,-1]} | cut -d'=' -f 1)=\"$(echo ${argvar[5,-1]} | cut -d'=' -f 2-)\"
    fi
done



# Update the style variables according to the state of no_style.
if [ ${no_style} -ne 0 ]
then
    error_style=$'\e[1;31m'
    warning_style=$'\e[33m'
    input_style=$'\e[1;34m'
    default_style=$'\e[0m'
fi
