## GET_ATTR_VAL

Get the value for an attribute of one object / instance.



### Syntax

```AdoScript
CC "Core" GET_ATTR_VAL objid:idValue ( attrid:idValue | attrname:strValue )
    [ as-string:boolValue ] [ core-value:boolValue ]
    [ format:strValue ] [ sep:strValue ]
# --> RESULT ecode:intValue val:anyValue
```



### Parameters

* `objid` (idValue) - The ID of the object for which the value should be
  retrieved.
* `attrid` (idValue) - The ID of the attribute whose value should be retrieved.
* `attrname` (strValue) - The name of the attribute whose value should be
  retrieved.
* `as-string` (boolValue, optional) - When true, then the value is always
  returned as its string representation. The default is false.
* `core-value` (boolValue, optional) - When true, then the internal content of
  the attribute is returned. Otherwise the result is the UI representation of
  the attribute value. The default is false.
* `format` (strValue, optional) - Specifies the format of the returned string
  for INTERREF type attributes.
* `sep` (strValue, optional) - Specifies a single character to use as the
  separator when an INTERREF type attribute has multiple entries. The default
  is " ".



### Returns

* `ecode` (intValue) - Contains the error code or 0 in case of success.
* `val` (anyValue) - The value of the attribute. It's type depends on the
  attribute's type and the used parameters.



### Remarks

Either `attrid` or `attrname` have to be provided as the parameter.

This command is mainly used to access the content of instances on "model"
level, but it can also be used to access content on "meta-model" level. This is
possible, because the object / instance provided by ID can be a modeling
object, a connector, a record row, an attribute profile, a library, a
(modeling) class, a relation class, a model etc. See [Examples](#examples)
section for how it all links together.

Accessing an attribute value will only work if the model / library is loaded at
the moment when the command is called. 

When `as-string` is specified, then `val` will be a string representation of
the attribute's value, no matter the attribute's type. Otherwise `val` is of
type:
* intValue for INTEGER attributes.
* realValue for DOUBLE attributes.
* timeValue for TIME attributes.
* strValue for all other types.

The `core-value` parameter can be used to access the value as it is specified
internally. For EXPRESSION attributes, the UI representation is the last
evaluated value, while the internal value is needed where an attribute value
shall be copied to another instance or be exported to a format which allows a
later reimport. Another difference is with INTERREF type attributes, where the
internal value is the LEO string, while the UI representation has a more user
friendly string.

The `format` and `sep` parameters can be used with INTERREF type attributes.
The `format` specifies how the returned string for each reference should look.
It can use the following placeholders to replace with the actual values:
* `%o` for object name
* `%c` for class name
* `%m` for model base name and model version
* `%t` for model type
* `%M` for model base name and model version (?)
* `%v` for model version
* `%V` for model version (long?)

With `sep` a single character can be specified to separate multiple entries in
the returned attribute value. Both `format` and `sep` parameters are ignored
when `core-value` is true.



### Examples

Get the value of the "Description" attribute of the first object from the
currently selected list. This example assumes that the model is already loaded.

```
# Get the ID of the first object from the currently selected list.
CC "Modeling" GET_SELECTED
SETL idFirstSelected:(VAL token(objids, 0))
# Get the object's "Description" attribute value.
CC "Core" GET_ATTR_VAL objid:(idFirstSelected) attrname:("Description")
SETL sDescription:(val)
# Show the "Description".
CC "AdoScript" VIEWBOX text:(sDescription)
```

Get the names of all selected objects. This example assumes that the model is
already loaded.

```AdoScript
# Get all selected objects.
CC "Modeling" GET_SELECTED
IF (objids = "") {
  CC "AdoScript" ERRORBOX "No object has been selected!"
  EXIT
}
SETL sSelectedIds:(objids)

SETL sSelectedNames:("")
# Iterate through all IDs
FOR sSelected in:(sSelectedIds) {
  # Get the class of the selected object.
  CC "Core" GET_CLASS_ID objid:(VAL sSelected)
  SETL idSelectedClass:(classid)

  # Get the attribute "Name" of the class.
  CC "Core" GET_ATTR_ID classid:(idSelectedClass) attrname:"Name"
  IF (ecode) {
    CC "AdoScript" ERRORBOX "The selected object does not contain an attribute called \"Name\"!"
    EXIT
  }
  SETL idNameAttr:(attrid)

  # Get the name of the selected object and add it to the result.
  CC "Core" GET_ATTR_VAL objid:(VAL sSelected) attrid:(idNameAttr)
  IF (ecode) {
    CC "AdoScript" ERRORBOX "Could not get the attribute value!"
    EXIT
  }
  SETL sSelectedNames:(tokcat(sSelectedNames, val, "\n"))
}

# Show the names of all selected elements.
CC "AdoScript" VIEWBOX text:(sSelectedNames)
```

Get the "Description" attribute value of the currently active dynamic model.

```AdoScript
# Get the current active model.
CC "Modeling" GET_ACT_MODEL
IF (modelid < 0) {
  CC "AdoScript" ERRORBOX "Open a dynamic model first!"
  EXIT
}
SETL idActiveModel:(modelid)

# Get the model attribute "Description".
CC "Core" GET_ATTR_ID classid:bp-model attrname:"Description"
IF (ecode) {
  CC "AdoScript" ERRORBOX "Model does not contain an attribute called \"Description\"!"
  EXIT
}
SETL idDescriptionAttr:(attrid)

# Get the model attribute.
# A simpler alternative would use attrname instead of attrid.
CC "Core" GET_ATTR_VAL objid:(idActiveModel) attrid:(idDescriptionAttr)
IF (ecode) {
  CC "AdoScript" ERRORBOX "Could not get the attribute value!"
  EXIT
}
SETL sDescription:(val)

# Show the "Description" of the active model.
CC "AdoScript" VIEWBOX text:(sDescription)
```

Get the default value of the "TestClass" class's "GraphRep" attribute.

```AdoScript
# Get the IDs of the currently loaded library.
CC "Core" GET_CURRENT_LIBRARIES
SETL idCurBpLib:(bplibid)
# Get the class ID of the class that "owns" the attribute.
CC "Core" GET_CLASS_ID classname:("TestClass")
SETL idClass:(classid)
# Get the default value of the "GraphRep" attribute.
CC "Core" GET_ATTR_VAL objid:(idClass) attrname:("GraphRep")
# Show the default "GraphRep" value.
CC "AdoScript" VIEWBOX text:(val)
```

Get the default value of the "Description" model attribute.

```AdoScript
# Get the class of the selected object.
CC "Core" GET_CLASS_ID classname:"__ModelTypeMetaData__" bp-library
SETL idMTClass:(classid)

# Get the attribute "Description" of the class.
CC "Core" GET_ATTR_ID classid:(idMTClass) attrname:"Description"
IF (ecode) {
  CC "AdoScript" ERRORBOX "The selected class does not contain an attribute called \"Description\"!"
  EXIT
}
SETL idMTDescription:(attrid)

# Get the default value for the "Description" attribute.
CC "Core" GET_ATTR_VAL objid:(idMTClass) attrid:(idMTDescription)
IF (ecode) {
  CC "AdoScript" ERRORBOX "Could not get the attribute value!"
  EXIT
}
SETL sDefaultDescription:(val)

# Show the default value of "Description".
CC "AdoScript" VIEWBOX text:(sDefaultDescription)
```

Get the dynamic library's "Modi" attribute value.

```AdoScript
# Get the IDs of the currently loaded library.
CC "Core" GET_CURRENT_LIBRARIES
SETL idCurBpLib:(bplibid)
# Get the class ID of the library that "owns" the "Modi" attribute.
CC "Core" GET_CLASS_ID objid:(idCurBpLib)
SETL idCurBpLibClass:(classid)
# Get the attribute ID of "Modi"
CC "Core" GET_ATTR_ID classid:(idCurBpLibClass) attrname:("Modi")
SETL idModi:(attrid)
# Get the current value of "Modi", so that we can append to it.
CC "Core" GET_ATTR_VAL objid:(idCurBpLib) attrid:(idModi)
SETL sOldModiVal:(val)
# Get the "Modi" attribute value.
CC "Core" GET_ATTR_VAL objid:(idCurBpLib) attrid:(idModi)
SETL sModi:(val)
# Show the value of "Modi".
CC "AdoScript" VIEWBOX text:(sModi)
```



### See Also

* `"Core" SET_ATTR_VAL`
* `"Core" GET_FACET_VAL`
* `"Core" SET_FACET_VAL`
* `"Core" LOAD_MODEL`
* `"Core" SAVE_MODEL`
* `"Core" LOAD_LIBRARY`
* `"Core" SAVE_LIBRARY`



### Versions and Changes

Available since ADOxx 1.3
