## CREATE_ATTRIBUTE

Creates a new attribute in the specified class.



### Syntax

```AdoScript
CC "Core" CREATE_ATTRIBUTE libid:idValue classid:idValue
    attrname:strValue attrtype:strValue
    [ class-attr:boolValue ] [ meta-model:boolValue ]
# --> RESULT ecode:intValue attrid:idValue
```



### Parameters

* `libid` (idValue) - The ID of the library where the attribute should be
  created. Use the ID of the dynamic (bp), static (we) or application library,
  depending on where the "owner" (classid) is located.
* `classid` (idValue) - The ID of the class where the attribute should be
  created, i.e. the class "owning" of the attribute.
* `attrname` (strValue) - The name for the new attribute. If an attribute with
  the name is already used in that class, then an error code will be returned.
* `attrtype` (strValue) - The type of the new attribute. It has to UPPERCASE
  name of the available type. Examples for the available types are: `INTEGER`,
  `DOUBLE`, `STRING`, `DISTRIBUTION`, `TIME`, `ENUMERATION`, `ENUMERATIONLIST`,
  `LONGSTRING`, `PROGRAMCALL`, `INTERREF`, `EXPRESSION`, `RECORD`,
  `ATTRPROFREF`, `DATE`, `DATETIME`, `HTTP` and `CLOB`.
* `class-attr` (boolValue, optional) - When true, then the new attribute will
  be a class attribute. Otherwise it is an instance attribute. The default is
  false.
* `meta-model` (boolValue, optional) - Uncertain if this does anything right
  now except throw an assertion error in some cases (only in debug version).
  Original documentation: If specified, then the new attribute is added at a
  meta model class. Meta model classes exist independent of any library and can
  be identified in the class hierarchy component by having the text
  "(Metamodel)" appended to the actual class name. The default is false.



### Returns

* `ecode` (intValue) - Contains the error code or 0 in case of success.
* `attrid` (idValue) - The ID of the new attribute, if creation was
  successful.



### Remarks

What types are available for an attribute and can thus be used for `attrtype`
depends on the class the attribute is created for. For example relation classes
do not support attributes of type `INTERREF`.

The creation of the attribute will only work if the library is loaded at the
moment when the command is called. Furthermore the change will only be
persisted, if the library is saved afterwards using `"Core" SAVE_LIBRARY`.

It also appears that the user permissions ("Authorisation" setting of a user in
the Development Toolkit) are irrelevant for calling this command. Having "only"
access to the Modelling Toolkit seems to be enough. However, this command can
only be called, if the used license has the "class hierarchy management"
component enabled. Otherwise it returns an ecode of 5023.



### Examples

Create an attribute in an existing class of the dynamic library through the
Modelling Toolkit.

```AdoScript
# Get the IDs of the currently loaded library.
CC "Core" GET_CURRENT_LIBRARIES
SETL idCurBpLib:(bplibid)
# Get the ID of the class that should "own" the new attribute.
CC "Core" GET_CLASS_ID classname:("__D-construct__")
SETL idClass:(classid)
# Create the new attribute.
CC "Core" CREATE_ATTRIBUTE libid:(idCurBpLib)
    classid:(idClass)
    attrname:("MyTestAttr") attrtype:("STRING")
SETL idAttr:(attrid)
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
# Show the ID of the newly created attribute.
CC "AdoScript" INFOBOX ("ID of created attribute: " + STR idAttr)
```

Create an attribute in an existing class of the dynamic library through the
Development Toolkit.

```AdoScript
# Get the ID of the library to be modified.
CC "Core" GET_LIBRARY_ID libname:("ADOxx 1.8 Experimentation Library")
SETL idCurAppLib:(libid)
# Load the library, so that it can be modified.
CC "Core" LOAD_LIBRARY libid:(idCurAppLib)
SETL idCurBpLib:(bplibid)
# Get the ID of the class that should "own" the new attribute.
CC "Core" GET_CLASS_ID classname:("__D-construct__")
SETL idClass:(classid)
# Create the new attribute.
CC "Core" CREATE_ATTRIBUTE libid:(idCurBpLib)
    classid:(idClass)
    attrname:("MyTestAttr") attrtype:("STRING")
SETL idAttr:(attrid)
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
# Discard the library since we're done modifying it.
CC "Core" DISCARD_LIBRARY
# Show the ID of the newly created attribute.
CC "AdoScript" INFOBOX ("ID of created attribute: " + STR idAttr)
```

Create a new class in the dynamic library and add a new attribute through the
Modelling Toolkit.

```AdoScript
# Get the IDs of the currently loaded library.
CC "Core" GET_CURRENT_LIBRARIES
SETL idCurBpLib:(bplibid)
# Get the ID of the super class for the new class.
CC "Core" GET_CLASS_ID classname:("__D-construct__")
SETL idSuperClass:(classid)
# Create the new class.
CC "Core" CREATE_CLASS libid:(idCurBpLib)
    superid:(idSuperClass)
    classname:"MyTestClass"
SETL idClass:(classid)
# Create the new attribute with the new class being the "owner".
CC "Core" CREATE_ATTRIBUTE libid:(idCurBpLib)
    classid:(idClass)
    attrname:("MyTestAttr") attrtype:("STRING")
SETL idAttr:(attrid)
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
# Show the ID of the newly created class and attribute.
CC "AdoScript" INFOBOX ("ID of created class: " + STR idClass + "\n" +
    "ID of created attribute: " + STR idAttr)
```

Create a new class in the dynamic library and add a new attribute through the
Development Toolkit.

```AdoScript
# Get the ID of the library to be modified.
CC "Core" GET_LIBRARY_ID libname:("ADOxx 1.8 Experimentation Library")
SETL idCurAppLib:(libid)
# Load the library, so that it can be modified.
CC "Core" LOAD_LIBRARY libid:(idCurAppLib)
SETL idCurBpLib:(bplibid)
# Get the ID of the super class for the new class.
CC "Core" GET_CLASS_ID classname:("__D-construct__")
SETL idSuperClass:(classid)
# Create the new class.
CC "Core" CREATE_CLASS libid:(idCurBpLib)
    superid:(idSuperClass)
    classname:"MyTestClass"
SETL idClass:(classid)
# Create the new attribute with the new class being the "owner".
CC "Core" CREATE_ATTRIBUTE libid:(idCurBpLib)
    classid:(idClass)
    attrname:("MyTestAttr") attrtype:("STRING")
SETL idAttr:(attrid)
# Save the library to persist the changes.
CC "Core" SAVE_LIBRARY libid:(idCurBpLib)
# Discard the library since we're done modifying it.
CC "Core" DISCARD_LIBRARY
# Show the ID of the newly created class and attribute.
CC "AdoScript" INFOBOX ("ID of created class: " + STR idClass + "\n" +
    "ID of created attribute: " + STR idAttr)
```



### See Also

* `"Core" LOAD_LIBRARY`
* `"Core" SAVE_LIBRARY`
* `"Core" DISCARD_LIBRARY`
* `"Core" CREATE_CLASS`
* `"Core" SET_FACET_VAL`
* `"Core" SET_ATTR_VAL`



### Versions and Changes

Available since ADOxx 1.7
