## BeforeCutObject

This event is triggered by the model editor before an object is cut from a
model. An object here is a modelling instance or connector.



### Syntax

```AdoScript
ON_EVENT "BeforeCutObject" {
  # PARAMETERS: instid:idValue classid:idValue modelid:idValue submodel:idValue
  # EXIT: -3, -2, -1, 0, >0
}
```



### Parameters

* `instid` (idValue) - The ID of the object.
* `classid` (idValue) - The ID of the class the object belongs to.
* `modelid` (idValue) - The ID of the model where the object is located.
* `submodel` (boolValue) - Is true when cutting the object is done in the
  context of creating a sub-model.



### Exit values

Controls whether the specific object is cut or not. Objects that are not cut
are neither deleted from the model, nor are they part of the clipboard (values
to be pasted). If no objects are cut, then the clipboard will remain unchanged.

* `0`  - Normal behavior.
* `-1` - Abort without error.
* `-2` - Abort with error. Aborts the entire cut operation as a whole.
* `-3` - Cut the object, but skip the check for referencing objects.
* `>0` - Abort with core error code. Aborts the entire cut operation as a
  whole.



### Remarks

The `submodel` parameter describes the context in which the cutting is done. It
is a boolean flag (i.e either 1 or 0). The value will be 1 if cutting is
implicitly triggered within the creation of a sub-model and not explicitly by
the user by cutting an object.

That event is sent before any cut related actions are performed. Such actions
are, e.g the check if an object is still referenced by another object, etc.

When multiple object's are cut at once, then the event is sent for each single
object and a different reaction can be returned for each one of them. This
allows for an adaptation of the objects to be cut.

The behavior concerning a selection of several objects is as follows:
1. If for at least one of the objects the event exits with an error (>0 or -2),
  the whole cut action is cancelled.
2. If for some of the objects -1 is returned and for some 0 or -3, then the
  selection is adjusted accordingly and the usual cut behavior proceeds.

When -3 is returned for some objects, those objects will be cut, but the check
if these objects are still referenced by other objects will not be performed.



### Examples

Show a popup with the name and class name of each object that is being cut.

```AdoScript
ON_EVENT "BeforeCutObject" {
  SETL idCutObj:(instid)
  # Get the name of the object that is being cut.
  CC "Core" GET_OBJ_NAME objid:(idCutObj)
  IF (ecode) { EXIT 0 }
  SETL sCutObjectName:(objname)
  # Get the class name of the object that is being cut.
  CC "Core" GET_CLASS_ID objid:(idCutObj)
  SETL idCutClass:(classid)
  CC "Core" GET_CLASS_NAME classid:(idCutClass)
  SETL sCutClassName:(classname)
  # Show both names.
  CC "AdoScript" INFOBOX (sCutObjectName + "\n" + sCutClassName)
}
```

Don't cut objects whose name contains "directors cut".

```AdoScript
ON_EVENT "BeforeCutObject" {
  SETL idCutObj:(instid)
  # Get the name of the object that is being cut.
  CC "Core" GET_OBJ_NAME objid:(idCutObj)
  IF (ecode) {
    # Allow cutting object's whose name we can't find out.
    EXIT 0
  }
  SETL sCutObjectName:(objname)
  # Abort cutting the object if it's name contains "directors cut".
  IF (search(sCutObjectName, "directors cut", 0) >= 0) {
    EXIT -1
  }
}
```

Abort the entire cut operation when one of the object's name contains
"directors cut".

```AdoScript
ON_EVENT "BeforeCutObject" {
  SETL idCutObj:(instid)
  # Get the name of the object that is being cut.
  CC "Core" GET_OBJ_NAME objid:(idCutObj)
  IF (ecode) {
    # Allow cutting object's whose name we can't find out.
    EXIT 0
  }
  SETL sCutObjectName:(objname)
  # Abort cutting the object if it's name contains "directors cut".
  IF (search(sCutObjectName, "directors cut", 0) >= 0) {
    EXIT -2
  }
}
```



### See Also

* `"Modeling" CUT_SELECTED`
* `ON_EVENT "BeforeCopyObject"`
