## BeforeCopyObject

This event is triggered by the model editor before an object within a model is
copied. An object here is a modelling instance or connector.



### Syntax

```AdoScript
ON_EVENT "BeforeCopyObject" {
  # PARAMETERS: instid:idValue classid:idValue modelid:idValue submodel:idValue
  # EXIT: -2, -1, 0, >0
}
```



### Parameters

* `instid` (idValue) - The ID of the object.
* `classid` (idValue) - The ID of the class the object belongs to.
* `modelid` (idValue) - The ID of the model where the object is located.
* `submodel` (boolValue) - Is true when copying the object is done in the
  context of creating a sub-model.



### Exit values

Controls whether the specific object is copied or not. Objects that are not
copied are not placed as part of the clipboard (values to be pasted). If no
objects are copied, then the clipboard will remain unchanged.

* `0`  - Normal behavior.
* `-1` - Abort without error.
* `-2` - Abort with error. Aborts the entire copy operation as a whole.
* `>0` - Abort with core error code. Aborts the entire copy operation as a
  whole.



### Remarks

The `submodel` parameter describes the context in which the copying is done. It
is a boolean flag (i.e either 1 or 0). The value will be 1 if copying is
implicitly triggered within the creation of a sub-model and not explicitly by
the user by copying an object.

That event is sent before any copy related actions are performed.

When multiple object's are copied at once, then the event is sent for each
single object and a different reaction can be returned for each one of them.
This allows for an adaptation of the objects to be copied.

The behavior concerning a selection of several objects is as follows:
1. If for at least one of the objects the event exits with an error (>0 or -2),
  the whole copy action is cancelled.
2. If for some of the objects -1 is returned and for some 0, then the selection
  is adjusted accordingly and the usual copy behavior proceeds.

Note that this event is only triggered by issuing an explicit copy action and
not if internally a copy is done within "cut".

Note that this event is only triggered by issuing an explicit copy action. It
is not triggered in the following cases:
* An internal copy when cutting objects.
* Putting objects into a copy buffer with `CC "Core" FILL_COPYBUFFER`.
* ...



### Examples

Show a popup with the name and class name of each object that is being copied.

```AdoScript
ON_EVENT "BeforeCopyObject" {
  SETL idCopyObj:(instid)
  # Get the name of the object that is being copied.
  CC "Core" GET_OBJ_NAME objid:(idCopyObj)
  IF (ecode) { EXIT 0 }
  SETL sCopyObjectName:(objname)
  # Get the class name of the object that is being copied.
  CC "Core" GET_CLASS_ID objid:(idCopyObj)
  SETL idCopyClass:(classid)
  CC "Core" GET_CLASS_NAME classid:(idCopyClass)
  SETL sCopyClassName:(classname)
  # Show both names.
  CC "AdoScript" INFOBOX (sCopyObjectName + "\n" + sCopyClassName)
}
```

Don't copy objects whose name contains "directors copy".

```AdoScript
ON_EVENT "BeforeCopyObject" {
  SETL idCopyObj:(instid)
  # Get the name of the object that is being copied.
  CC "Core" GET_OBJ_NAME objid:(idCopyObj)
  IF (ecode) {
    # Allow copying object's whose name we can't find out.
    EXIT 0
  }
  SETL sCopyObjectName:(objname)
  # Abort copying the object if it's name contains "directors copy".
  IF (search(sCopyObjectName, "directors copy", 0) >= 0) {
    EXIT -1
  }
}
```

Abort the entire copy operation when one of the object's name contains
"directors copy".

```AdoScript
ON_EVENT "BeforeCopyObject" {
  SETL idCopyObj:(instid)
  # Get the name of the object that is being copied.
  CC "Core" GET_OBJ_NAME objid:(idCopyObj)
  IF (ecode) {
    # Allow copying object's whose name we can't find out.
    EXIT 0
  }
  SETL sCopyObjectName:(objname)
  # Abort copying the object if it's name contains "directors copy".
  IF (search(sCopyObjectName, "directors copy", 0) >= 0) {
    EXIT -2
  }
}
```



### See Also

* `"Modeling" COPY_SELECTED`
* `"Core" FILL_COPYBUFFER`
* `ON_EVENT "BeforeCutObject"`
